/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Copyright (C) 2011 The Reconfigurable Multi-resolutions Profiling Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package rmp;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.Vector;

public class ResultDatabase {
	private Vector<ResultData> resultDatabase;
	private boolean havetimeResultFile;
	private boolean haveenergyResultFile;
	//private Vector<String> cleanList;
	private int logSize;
	private double energy_consumption;
	private double base_time;
	
	public ResultDatabase()
	{
		havetimeResultFile = false;
		haveenergyResultFile = false;
		logSize = 0;
		energy_consumption = 0.0;
		base_time = 0.0;
	}
	
	public int update(ProbeTableMode pTableMode, AppCallerTableMode appTableMode, FrameCallerTableMode frameTableMode)
	{
		Vector<ResultData> app_caller_energy_data = new Vector<ResultData>();
		Vector<ResultData> frame_caller_energy_data = new Vector<ResultData>();
		ResultData temp = null;
		boolean caller_mode = false;
		
		//Store last result for energy analysis
		if (appTableMode.getRowCount() != 0 || frameTableMode.getRowCount() != 0)
		{
			caller_mode = true;
			//app
			for(int i=0;i<appTableMode.getRowCount();i++)
			{
				temp = new ResultData();
				for (int j=0;j<pTableMode.getRowCount();j++)
				{
					if (pTableMode.getValueAt(j, 0).toString().equals(appTableMode.getValueAt(i, 0).toString()))
					{
						temp.setName(pTableMode.getValueAt(j, 0).toString());
						temp.setEnergyResult(Double.parseDouble(pTableMode.getValueAt(j, 4).toString()));
						app_caller_energy_data.add(temp);
						break;
					}
				}				
			}
			//frame
			for(int i=0;i<frameTableMode.getRowCount();i++)
			{
				temp = new ResultData();
				for (int j=0;j<pTableMode.getRowCount();j++)
				{
					if (pTableMode.getValueAt(j, 0).toString().equals(frameTableMode.getValueAt(i, 0).toString()))
					{
						temp.setName(pTableMode.getValueAt(j, 0).toString());
						temp.setEnergyResult(Double.parseDouble(pTableMode.getValueAt(j, 4).toString()));
						frame_caller_energy_data.add(temp);
						break;
					}
				}				
			}
		}
		
		ResultContent result_content = new ResultContent();
		ResultData result_data = null;
		String time_data = null, energy_data = null;
		String[] element_time = null;
		long result_time = 0;
		
		BufferedReader timeResultFile = null;
		BufferedReader energyResultFile = null;
		
		logSize = 0;
		// Open the result logs
		try {
			timeResultFile = new BufferedReader(new FileReader("timeResult.txt"));
			havetimeResultFile = true;
			File file = new File("timeResult.txt");
			logSize += file.length();
		} catch (FileNotFoundException e) {
			havetimeResultFile = false;
		}
		
		try {
			energyResultFile = new BufferedReader(new FileReader("energyResult.txt"));
			haveenergyResultFile = true;
		} catch (FileNotFoundException e) {
			haveenergyResultFile = false;
		}
		
		resultDatabase = new Vector<ResultData>();
		// Read the time result from logs
		if (havetimeResultFile == true)
		{
			try {
				while((time_data = timeResultFile.readLine()) != null)
				{
					element_time = time_data.split(" ");
					result_data = new ResultData();
					if (element_time.length > 3)
						result_data.setNoLog(Integer.parseInt(element_time[3]));
					else if (element_time[0].startsWith("-"))
					{
						element_time[0] = element_time[0].substring(1);
						if (result_content.stackFirstElement() != null)
							result_content.stackFirstElement().setNoLog(result_content.stackFirstElement().getNoLog() + Integer.parseInt(element_time[0]));
						continue;
					}
					result_data.setName(element_time[0]);
					result_time = Long.parseLong(element_time[1]) * 1000000000 + Long.parseLong(element_time[2]);
					result_data.setInclusiveResult(result_time);
					
					if (result_content.push(result_data))
						resultDatabase.add(result_content.pop());
				}
				timeResultFile.close();
	
			} catch (IOException e) {
				e.printStackTrace();
			}
			
			//calculate exclusive time
			for (int i=0;i<this.getSize();i++)
			{
				Vector<Integer> child = this.getDataFromIndex(i).getChild();
				long exclusive_result = this.getDataFromIndex(i).getInclusiveResult();
				
				for (int j=0;j<child.size();j++)
					exclusive_result -= this.getData(child.elementAt(j)).getInclusiveResult();

				this.getDataFromIndex(i).setExclusiveResult(exclusive_result);
			}
			
			//minus no log inclusive results
			for (int i=0;i<this.getSize();i++)
			{
				Vector<Integer> child = this.getDataFromIndex(i).getChild();
				long real_inclusive_result = this.getDataFromIndex(i).getInclusiveResult();
				int sum_nolog = this.getDataFromIndex(i).getNoLog();
				
				//System.out.printf("No log start %d at %s\n", sum_nolog, this.getDataFromIndex(i).getName());
				for (int j=0;j<child.size();j++)
					sum_nolog += this.getChildNoLogRecursive(child.elementAt(j));
				
				this.getDataFromIndex(i).setRealInclusiveResult(real_inclusive_result - (sum_nolog * 583333));
				//System.out.printf("No log end: %d\n", sum_nolog);
			}
			
			//minus no log exclusive results
			for (int i=0;i<this.getSize();i++)
			{
				long real_exclusive_result = this.getDataFromIndex(i).getExclusiveResult();
				int nolog = this.getDataFromIndex(i).getNoLog();
				
				//System.out.printf("No log start %d at %s\n", nolog, this.getDataFromIndex(i).getName());
				this.getDataFromIndex(i).setRealExclusiveResult(real_exclusive_result - (nolog * 583333));
			}
		}
		// Read the energy result from logs
		if (haveenergyResultFile == true)
		{	
			if (caller_mode == true)
			{
				for (int i=0;i<app_caller_energy_data.size();i++)
					for (int j=0;j<this.getSize();j++)
						if (app_caller_energy_data.elementAt(i).getName().equals(this.getDataFromIndex(j).getName()))
						{
							double e_temp = app_caller_energy_data.elementAt(i).getEnergyResult();
							//this.getDataFromIndex(j).setEnergyResult(app_caller_energy_data.elementAt(i).getEnergyResult());
							this.getDataFromIndex(j).setEnergyResult(e_temp);
						}
				for (int i=0;i<frame_caller_energy_data.size();i++)
					for (int j=0;j<this.getSize();j++)
						if (frame_caller_energy_data.elementAt(i).getName().equals(this.getDataFromIndex(j).getName()))
							this.getDataFromIndex(j).setEnergyResult(frame_caller_energy_data.elementAt(i).getEnergyResult());
				this.energy_consumption = -1;
			}
			else
			{
				try {
					if((energy_data = energyResultFile.readLine()) != null)
						this.energy_consumption = Double.parseDouble(energy_data);
					energyResultFile.close();
		
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}		
		return this.logSize;
	}
	
	public void calEnergy()
	{
		double sum_time = 0;
		long head_time = -1;
		long block_result = 0;
		double real_energy = 0.0;
		
		if (this.energy_consumption != -1)
		{
			// Get sum of all time data
			for (int i=0;i<resultDatabase.size();i++)
			{
				if (resultDatabase.get(i).getCaller() == -1)
				{
					if (head_time != -1)
						block_result += resultDatabase.get(i).getHead() - head_time;
					head_time = resultDatabase.get(i).getTail();
				}
				sum_time += resultDatabase.get(i).getRealExclusiveResult();
			}
			sum_time += block_result;
			base_time = sum_time;
			
			// Separate energy data to all probes
			for (int i=0;i<resultDatabase.size();i++)
			{
				double probe_temp_time = resultDatabase.get(i).getRealExclusiveResult();
				double proportion = probe_temp_time / sum_time;
				resultDatabase.get(i).setEnergyResult(energy_consumption * proportion);
			}
		}
		else
		{
			for (int i=0;i<resultDatabase.size();i++)
			{
				if (resultDatabase.get(i).getEnergyResult() > 0)
				{
					//Get all child's time to be the sum time
					Vector<Integer> child = resultDatabase.get(i).getChild();
					for (int j=0;j<child.size();j++)
						sum_time += getChildSumTime(child.elementAt(j));
					sum_time += resultDatabase.get(i).getRealExclusiveResult();
					//Write real energy to each child 
					for (int j=0;j<child.size();j++)
						setChildEnergy(resultDatabase.get(i).getEnergyResult(), sum_time, child.elementAt(j));
					//self
					resultDatabase.get(i).setEnergyResult(resultDatabase.get(i).getEnergyResult() * (resultDatabase.get(i).getRealExclusiveResult()/sum_time));
				}
			}
		}
	}
	
	public double calBlockEnergyResult(double time)
	{
		double proportion = time / base_time;
		return energy_consumption * proportion;
	}
	
	public boolean haveTimeResult()
	{
		return this.havetimeResultFile;
	}
	
	public boolean haveEnergyResult()
	{
		return this.haveenergyResultFile;
	}
	
	public int getSize()
	{
		return this.resultDatabase.size();
	}

	public ResultData getDataFromIndex(int index)
	{
		return this.resultDatabase.get(index);
	}
	
	public int getDataIndex(int id)
	{
		for(int i=0;i<this.getSize();i++)
		{
			if (this.resultDatabase.get(i).getID() == id)
				return i;
		}
		return -1;
	}
	
	public ResultData getData(int id)
	{
		for(int i=0;i<this.getSize();i++)
		{
			if (this.resultDatabase.get(i).getID() == id)
				return this.resultDatabase.get(i);
		}
		return null;
	}
	
	public Vector<ResultData> getDatas(String name)
	{
		Vector<ResultData> rtnData = new Vector<ResultData>();
		
		for(int i=0;i<this.getSize();i++)
		{
			if (this.resultDatabase.get(i).getName().equals(name))
				rtnData.add(this.resultDatabase.get(i));
		}
		
		return rtnData;
	}
	
	public long getSumTime()
	{
		long sum = 0;
		
		for (int i=0;i<this.getSize();i++)
			sum += this.resultDatabase.get(i).getExclusiveResult();
		
		return sum;
	}

	public int getChildNoLogRecursive(int id)
	{
		int ans = 0;
		
		for(int i=0;i<this.getSize();i++)
		{
			if (this.resultDatabase.get(i).getID() == id)
			{
				//if (this.getDataFromIndex(i).getNoLog() != 0)
				//	System.out.printf("Find %d at %s\n", this.getDataFromIndex(i).getNoLog(), this.getDataFromIndex(i).getName());
				ans = this.getDataFromIndex(i).getNoLog();
				Vector<Integer> child = this.getDataFromIndex(i).getChild();
				for (int j=0;j<child.size();j++)
					ans += getChildNoLogRecursive(child.elementAt(j));
				break;
			}
		}
		return ans;
	}
	
	public void collectChildName(int id, Vector<String> name)
	{
		for(int i=0;i<this.getSize();i++)
		{
			if (this.resultDatabase.get(i).getID() == id)
			{
				name.add(this.resultDatabase.get(i).getName());
				Vector<Integer> child = this.getDataFromIndex(i).getChild();
				for (int j=0;j<child.size();j++)
					collectChildName(child.elementAt(j), name);
				break;
			}
		}
	}
	
	public double getChildSumTime(int id)
	{
		double ans = 0.0;
		for(int i=0;i<this.getSize();i++)
		{
			if (this.resultDatabase.get(i).getID() == id)
			{
				ans += this.resultDatabase.get(i).getRealExclusiveResult();
				Vector<Integer> child = this.resultDatabase.get(i).getChild();
				for (int j=0;j<child.size();j++)
					ans += getChildSumTime(child.elementAt(j));
				break;
			}
		}
		
		return ans;
	}
	
	public void setChildEnergy(double energy, double sum_time, int id)
	{
		long child_time = 0;
		double real_energy = 0.0;
		for(int i=0;i<this.getSize();i++)
		{
			if (this.resultDatabase.get(i).getID() == id)
			{
				child_time = this.resultDatabase.get(i).getRealExclusiveResult();
				real_energy = energy * (child_time/sum_time);
				this.resultDatabase.get(i).setEnergyResult(real_energy);
				Vector<Integer> child = this.resultDatabase.get(i).getChild();
				for (int j=0;j<child.size();j++)
					setChildEnergy(energy, sum_time, child.elementAt(j));
				break;
			}
		}
	}
}
