/*
 * setup_pcap_info.c
 * Code for constructing the information of PCAP from database "pcap_lib" to database "training_data". 
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <mysql.h>
#include <sys/types.h> /* opendir, lstat */
#include <sys/stat.h> /* lstat */
#include <dirent.h> /* opendir, readdir */
#include <unistd.h> /* lstat */
#include <err.h>

#include "setup_pcap_info.h"

static struct Pcap pcaps[MAXSIZE];


int main(int argc, char **argv)
{
	DIR *fp;
	struct dirent *ent;
	struct stat sb;

	/*
	 * Get and parse command-line options
	 */
	int opt;
	char *processed_directory;

	opt = 0;
	processed_directory = NULL;
	while((opt = getopt(argc, argv, "d:")) != -1) {
		switch(opt) {
			case 'd':
				processed_directory = optarg;
				break;
			default: /* '?' */
				fprintf(stderr, "Usage: %s [-d Processed_directory]\n", argv[0]);
				exit(0);
		}
	}

	/* Instead of "if(optind >= argc)", because the getopt() does not set the initial value of optind to '0'. */
	if(optind == 1) {
		fprintf(stderr, "Expected argument after options\n");
		exit(0);
	}
	if(processed_directory == NULL) {
		printf("Error: Processed directory should be set.\n");
		exit(0);
	}


	chdir(processed_directory);
	
	if((fp = opendir(processed_directory)) == NULL) { 
		exit(0); 
	}

	while((ent = readdir(fp)) != NULL) {
		if(lstat(ent->d_name, &sb) == -1) { 
			err(1, "lstat"); 
		}
		if(S_ISDIR(sb.st_mode)) {
			continue;
		}
		if(strcmp(ent->d_name, ".")==0 || strcmp(ent->d_name, "..")==0) {
			continue;
		}
		setup_pcap_info(ent->d_name);
	}

	return 0;
}

void setup_pcap_info (char *file_name)
{
	int i, j;

	/* Preprocessing to set up the pcap_id data */
	{
		pre_pcap_info(file_name, pcaps);
		for(i=0; i<MAXSIZE && pcaps[i].pcap_id[0]!=0; ++i) {
			printf("pcap_id[%d]=%s\n", i, pcaps[i].pcap_id);
		}
	}
	
	MYSQL *conn_pcaplib, *conn_training;
	conn_pcaplib = mysql_init(NULL);
	conn_training = mysql_init(NULL);

	if(conn_pcaplib == NULL) {
		printf("Error %u: %s\n", mysql_errno(conn_pcaplib), mysql_error(conn_pcaplib));
		exit(1);
	}
	if(conn_training == NULL) {
		printf("Error %u: %s\n", mysql_errno(conn_training), mysql_error(conn_training));
		exit(1);
	}

	/* Connect to database "pcap_lib" */
	if(mysql_real_connect(conn_pcaplib, SRC_HOST, SRC_USER, SRC_PASSWORD, SRC_DB, 0, NULL, 0 ) == NULL) {
		printf("Error %u: %s\n", mysql_errno(conn_pcaplib), mysql_error(conn_pcaplib));
		exit(1);
	}
	/* Connect to database "training_data" */
    if(mysql_real_connect(conn_training, DST_HOST, DST_USER, DST_PASSWORD, DST_DB, 0, NULL, 0 ) == NULL) {
        printf("Error %u: %s\n", mysql_errno(conn_training), mysql_error(conn_training));
        exit(1);
	}

	/* Select from database "pcap_lib" and insert into database "training_data" */
	{
		MYSQL_RES *result;
		MYSQL_ROW row;
		int num_fields;
		char cmd[MAXLINE], insert_cmd[MAXLINE];
		char buf[MAXLINE];

		memset(cmd, 0, MAXLINE);
		for(i=0; i<MAXSIZE && pcaps[i].pcap_id[0]!=0; ++i) {
			sprintf(cmd, "SELECT t2.pcap_id, t2.msg, t4.dev_name, t5.timestamp, t3.ver_id, t3.firmware_ver, t3.signature_ver FROM syslog as t2, verInfo as t3, devInfo as t4, pcapInfo as t5 WHERE t2.ver_id=t3.ver_id AND t3.dev_id=t4.dev_id AND t2.pcap_id=t5.pcap_id AND t2.pcap_id=%s AND t2.msg <> 'NULL'", pcaps[i].pcap_id);
			mysql_query(conn_pcaplib, cmd);
			result = mysql_store_result(conn_pcaplib);
			num_fields = mysql_num_fields(result);
			while((row = mysql_fetch_row(result)) != NULL) {
				memset(buf, 0, sizeof(buf));
				for(j=0; j<num_fields; ++j) {
					printf("%s | " , row[j]);				
					if(j == 0) strncat(buf, "'", 1);
					strcat(buf, row[j]);
					strncat(buf, "'", 1);
					if(j != (num_fields-1)) strncat(buf, ", '", 3);
				}
				printf("\n");
				memset(insert_cmd, 0, MAXLINE);
				sprintf(insert_cmd, "INSERT INTO pcapInfo(pcap_id, msg, dev_name, timestamp, ver_id, firmware_ver, signature_ver) VALUES(%s)", buf);
				printf("insert_cmd=%s\n", insert_cmd);
				mysql_query(conn_training, insert_cmd);			
			}
			printf("===============================================================\n");
			mysql_free_result(result);
		}
	}

	mysql_close(conn_pcaplib);
	mysql_close(conn_training);
}

void pre_pcap_info (char *file_path, struct Pcap *pcap_data)
{
	int i;
	char buf[MAXLINE];
	FILE *fp;

	fp = fopen(file_path, "r");
	if(fp == NULL) exit(0);

	memset(pcap_data, 0, sizeof(struct Pcap)*MAXSIZE);

	/* Extract Pcap ID from pcap name, e.g, extract XXXXXXXXXX from XXXXXXXXXX.pcap */
	for(i=0; i<MAXSIZE && fgets(buf, MAXLINE, fp)!=NULL; ++i) {
		char *p;

		p = strtok(buf, ".");
		if(p == NULL) { printf("Error: This should not happen\n"); exit(0); }
		strncpy(pcap_data[i].pcap_id, p, sizeof(pcap_data[i].pcap_id));
	}
	fclose(fp);
}

