/*
 * cwv.c
 * Code for Creditability-based Weighted Voting algorithm to accurately decide the trace
 * one by one by leveraging multiple IDSs' domain knowledge.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <mysql.h>

#include "cwv.h"

static struct Trace training_traces, traces;
static struct Alert training_data[MAXSIZE], data[MAXSIZE];
static struct Vendor vendors[NUMBER_OF_DEVICES];


int main(int argc, char **argv)
{
	int i, j;
	
	//printf("size of Alert: %d\n", sizeof(data)+sizeof(training_data));
	//exit(0);
	//printf("Detection threshold=%f\n", DETECTION_THRESHOLD);
	//printf("Abnormal threshold=%f\n", ABNORMALITY_THRESHOLD);
	//printf("ALPHA=%f\n", ALPHA);
	//printf("BETA=%f\n", BETA);

	/* 
	 * Get and parse command-line options 
	 */
	int opt;
	int mv_flag; /* Switch flag between "Creditability-based Weighted Voting" (default) and "Majority Voting" */
	char *training_data_file, *processing_data_file;
	char *db_host, *db_user, *db_password, *db_db; /* for database connection */

	opt = 0;
	mv_flag = 0;
	training_data_file = NULL;
	processing_data_file = NULL;
	db_host = db_user = db_password = db_db = NULL;
	while((opt = getopt(argc, argv, "T:P:H:u:p:D:mh")) != -1) {
		switch(opt) {
			case 'T':
				training_data_file = optarg;
				break;
			case 'P':
				processing_data_file = optarg;
				break;
			case 'H':
				db_host = optarg;
				break;
			case 'u':
				db_user = optarg;
				break;
			case 'p':
				db_password = optarg;
				break;
			case 'D':
				db_db = optarg;
				break;
			case 'm':
				mv_flag = 1;
				break;
			case 'h':
				help_msg();
				exit(0);
			default: /* '?' */
				fprintf(stderr, "Usage: %s [-T Training_data] [-P Processing_data] [-H DB_host] [-u DB_user] [-p DB_password] [-D DB_database]\n", argv[0]);
				exit(0);
		}
	}
	//printf("Training=%s; Processing=%s; optind=%d; argc=%d\n", training_data_file, processing_data_file, optind, argc);

	/* Instead of "if(optind >= argc)", because the getopt() does not set the initial value of optind to '0'. */
	if(optind == 1) {
		fprintf(stderr, "Expected argument after options\n");
		exit(0);
	}
	if(training_data_file == NULL || processing_data_file == NULL) {
		printf("Error: Both training and processing data should be set.\n");
		exit(0);
	}
	if(db_host == NULL || db_user == NULL || db_password == NULL || db_db == NULL) {
		printf("Error: All variables for MySQL should be set.\n");
		exit(0);
	}


	/*
	 * Preprocessing to set up the training data
	 */
	{
		//pre_alert_info(training_data_file, training_data);
		//pre_trace_info(training_data, &training_traces);
		pre_pcap_id_from_file(training_data_file, training_data);
        pre_alert_from_db(training_data, db_host, db_user, db_password, db_db);
		pre_trace_info(training_data, &training_traces);
		pre_vendor_info();
		
		printf("========= Training Data: Alert Information Table =================\n");
		for(i=0; i<MAXSIZE && training_data[i].pcap_id[0]!=0; ++i) {
			printf("pcap_id=%s", training_data[i].pcap_id);
			for(j=0; j<NUMBER_OF_DEVICES; ++j) {
				printf(" [%d]", training_data[i].alert[j]);
			}
			printf(" confirm=%d\n", training_data[i].confirm);
		}
		printf("==================================================================\n");
		printf("Number of training traces: %d\n", training_traces.num_pcaps);
	    printf("Number of malicious training traces: %d\n", training_traces.num_mali);
	    printf("Number of healthy training traces: %d\n", training_traces.num_pcaps-training_traces.num_mali);
	
	    for(i=0; i<NUMBER_OF_DEVICES; ++i) {
	        printf("Vendor %d: TP=%f FP=%f TN=%f FN=%f\t", i, vendors[i].TP, vendors[i].FP, vendors[i].TN, vendors[i].FN);
	        printf("P(I|A)=%f P(-I|-A)=%f\n", vendors[i].weight[0], vendors[i].weight[1]);
	    }
	}

	/*
	 * Preprocessing to set up the processing data
	 */
	{
		//pre_alert_info(processing_data_file, data);
		//pre_trace_info(data, &traces);
		pre_pcap_id_from_file(processing_data_file, data);
		pre_alert_from_db(data, db_host, db_user, db_password, db_db);
		pre_trace_info(data, &traces);
		
		printf("========= Processing Data: Alert Information Table ===============\n");
		for(i=0; i<MAXSIZE && data[i].pcap_id[0]!=0; ++i) {
			printf("pcap_id=%s", data[i].pcap_id);
			for(j=0; j<NUMBER_OF_DEVICES; ++j) {
				printf(" [%d]", data[i].alert[j]);
			}
			printf(" confirm=%d\n", data[i].confirm);
			for(j=0; j<NUMBER_OF_DEVICES; ++j) {
				printf("data[%d].msg[%d]= %s, data[%d].dev_name[%d]= %s\n" , i, j, data[i].msg[j], i, j, data[i].dev_name[j]);
	        }
		}
		printf("==================================================================\n");
		printf("Number of processing traces: %d\n", traces.num_pcaps);
		printf("Number of malicious processing traces: %d\n", traces.num_mali);
		printf("Number of healthy processing traces: %d\n", traces.num_pcaps-traces.num_mali);
	}

	/* Switch tthe Majority Voting algorithm */
	if(mv_flag == 1) {
		majority_voting();
	}

	/* Use Creditability-based Weighted Voting algorithm (default) */
	else {
		/*
		 * Authority Selecting
		 */
		{
			int authority_exist = 0;

			authority_selecting();
			for(i=0; i<NUMBER_OF_DEVICES; ++i) {
				printf("vendors[%d].authority=%d\n", i, vendors[i].authority);
				if(vendors[i].authority == 1) {
					authority_exist++;
				}
			}

			/* If authority exist, we decide directly according to it */
			if(authority_exist > 0) {
				direct_decision();
				exit(0);
			}
		}

		/*
		 * Voter Excluding 
		 */
		{
			int num_excluded_voter = 0;

			voter_excluding();
			for(i=0; i<NUMBER_OF_DEVICES; ++i) {
				printf("vendor[%d].excluded=%d\n", i, vendors[i].excluded);
				printf("Vendor %d: P(I|A)=%f P(-I|-A)=%f\n", i, vendors[i].weight[0], vendors[i].weight[1]);
				if(vendors[i].excluded == 1) {
					num_excluded_voter++;
				}
			}

			/* If there is only one voter, we set it to be an authority */
			if(num_excluded_voter == NUMBER_OF_DEVICES-1) {
				for(i=0; i<NUMBER_OF_DEVICES; ++i) {
					if(vendors[i].excluded == 0) {
						vendors[i].authority = 1;
					}
				}
				direct_decision();
				exit(0);
			}
		}

		/*
		 * Weighted Voting
		 */
		{
			int num_excluded_voters = 0;
			for(i=0; i<NUMBER_OF_DEVICES; ++i) {
				if(vendors[i].excluded == 1) {
					num_excluded_voters++;
				}
			}
			weighted_voting(NUMBER_OF_DEVICES-num_excluded_voters, db_host, db_user, db_password, db_db);
		}
	}

	printf("=============== Result: Alert Information Table ========================\n");
	for(i=0; i<MAXSIZE && data[i].pcap_id[0]!=0; ++i) {
		printf("pcap_id=%s", data[i].pcap_id);
		for(j=0; j<NUMBER_OF_DEVICES; ++j) {
			printf(" [%d]", data[i].alert[j]);
		}
		printf(" confirm=%d", data[i].confirm);
		printf(" result= %d\n", data[i].v_result);
	}
	printf("========================================================================\n");

	return 0;
}

/*
 * Function: pre_pcap_id_from_file
 * Preprocessing: Set up pcap_id information.
 * Requirement:	Store pcap_id data into struct Alert data[i].pcap_id from file.
 */
void pre_pcap_id_from_file (char *file_path, struct Alert *alert_data)
{
	int i;
	char buf[MAXLINE];
	FILE *fp;

	fp = fopen(file_path, "r");
	if(fp == NULL) exit(0);

	memset(alert_data, 0, sizeof(struct Alert)*MAXSIZE);
	for(i=0; i<MAXSIZE && fgets(buf, MAXLINE, fp)!=NULL; ++i) {
		char *p;
		
		p = strtok(buf, ".");
		if(p == NULL) { printf("Error: Input should be XXXXXXXXXX.pcap\n"); exit(0); }
		strncpy(alert_data[i].pcap_id, p, sizeof(alert_data[i].pcap_id));
	}
	fclose(fp);
}

/*
 * Function: pre_alert_from_db
 * Requirements: According to pcap_id, set up msg, dev_name, data[i].alert[NUMBER_OF_DEVICES], data[i].confirm 
 */
void pre_alert_from_db (struct Alert *alert_data, char *db_host, char *db_user, char *db_password, char *db_db)
{
	int i, j;

	MYSQL *conn_training;
	conn_training = mysql_init(NULL);
	if(conn_training == NULL) {
		printf("Error %u: %s\n", mysql_errno(conn_training), mysql_error(conn_training));
		exit(1);
	}

	/* Connect to database "training_data" */
	if(mysql_real_connect(conn_training, db_host, db_user, db_password, db_db, 0, NULL, 0 ) == NULL) {
		printf("Error %u: %s\n", mysql_errno(conn_training), mysql_error(conn_training));
		exit(1);
	}

	MYSQL_RES *result;
	MYSQL_ROW row;
	char cmd[MAXLINE];

	for(i=0; i<MAXSIZE && alert_data[i].pcap_id[0]!=0; ++i) {
		memset(cmd, 0, MAXLINE);
		sprintf(cmd, "SELECT msg, dev_name, confirm FROM pcapInfo WHERE pcap_id='%s'", alert_data[i].pcap_id);
		mysql_query(conn_training, cmd);	
		result = mysql_store_result(conn_training);
		for(j=0; j<NUMBER_OF_DEVICES; ++j) {
			if((row = mysql_fetch_row(result)) == NULL) {
				break;
			}

			if(strcmp(row[1], "FortiGate 110c") == 0) {
				strncpy(alert_data[i].msg[0], row[0], sizeof(alert_data[i].msg[0])-1);
				strncpy(alert_data[i].dev_name[0], row[1], sizeof(alert_data[i].dev_name[0])-1);
				alert_data[i].alert[0] = 1;
			}
			else if(strcmp(row[1], "ZyxelWall 1050") == 0) {
				strncpy(alert_data[i].msg[1], row[0], sizeof(alert_data[i].msg[1])-1);
				strncpy(alert_data[i].dev_name[1], row[1], sizeof(alert_data[i].dev_name[1])-1);
				alert_data[i].alert[1] = 1;
			}
			else if(strcmp(row[1], "5000E") == 0) {
				strncpy(alert_data[i].msg[2], row[0], sizeof(alert_data[i].msg[2])-1);
				strncpy(alert_data[i].dev_name[2], row[1], sizeof(alert_data[i].dev_name[2])-1);
				alert_data[i].alert[2] = 1;
			}
			else if(strcmp(row[1], "TDA2") == 0) {
				strncpy(alert_data[i].msg[3], row[0], sizeof(alert_data[i].msg[3])-1);
				strncpy(alert_data[i].dev_name[3], row[1], sizeof(alert_data[i].dev_name[3])-1);
				alert_data[i].alert[3] = 1;
			}
			else if(strcmp(row[1], "DFL-1600") == 0) {
				strncpy(alert_data[i].msg[4], row[0], sizeof(alert_data[i].msg[4])-1);
				strncpy(alert_data[i].dev_name[4], row[1], sizeof(alert_data[i].dev_name[4])-1);
				alert_data[i].alert[4] = 1;
			}
			else if(strcmp(row[1], "NetKeeper7K") == 0) {
				strncpy(alert_data[i].msg[5], row[0], sizeof(alert_data[i].msg[5])-1);
				strncpy(alert_data[i].dev_name[5], row[1], sizeof(alert_data[i].dev_name[5])-1);
				alert_data[i].alert[5] = 1;
			}
			else if((strcmp(row[1], "M-1250")==0) || (strcmp(row[1], "IntruShield 1400")==0)) {
				strncpy(alert_data[i].msg[6], row[0], sizeof(alert_data[i].msg[6])-1);
				strncpy(alert_data[i].dev_name[6], row[1], sizeof(alert_data[i].dev_name[6])-1);
				alert_data[i].alert[6] = 1;
			} else {
				continue;
			}

			if(strcmp(row[2], "1") == 0) {
				alert_data[i].confirm = 1;
			} else {
				alert_data[i].confirm = 0;
			}
		}
		mysql_free_result(result);	
	}
	mysql_close(conn_training);
}

/*
 * Preprocessing: Set up alert information
 * Components: pcap_id, alert[NUMBER_OF_DEVICES], confirm
 */
int pre_alert_info (char *file_path, struct Alert *alert_data)
{
	int i, j;
	char buf[MAXLINE];
	FILE *fp;

	fp = fopen(file_path, "r");
	if(fp == NULL) return 0;

	memset(alert_data, 0, sizeof(struct Alert)*MAXSIZE);

	for(i=0; i<MAXSIZE && fgets(buf, MAXLINE, fp)!=NULL; ++i) {
		char *p;

		p = strtok(buf, ", \t");
		if(p == NULL) { printf("This should not happen\n"); return 0; }
		strncpy(alert_data[i].pcap_id, p, sizeof(alert_data[i].pcap_id));
		for(j=0; j<NUMBER_OF_DEVICES; ++j) {
			p = strtok(NULL, ", \t");
			if(p == NULL) { printf("This should not happen\n"); return 0; }
				alert_data[i].alert[j] = ((*p) == '0') ? 0 : 1;
		}
		p = strtok(NULL, ", \t");
		if(p == NULL) { printf("This should not happen\n"); return 0; }
		alert_data[i].confirm = ((*p) == '0') ? 0 : 1;
	}
	fclose(fp);

	return 1;
}

/*
 * Preprocessing: Set up traces information
 * Components: num_pcaps, num_mali
 */
int pre_trace_info (struct Alert *alert_data, struct Trace *trace_data)
{
	int i;
	
	memset(trace_data, 0, sizeof(struct Trace));

	/*
	 * '0' is a string terminator, alert_data[i].pcap_id[0]==0 means initial value
	 */
	for(i=0; i<MAXSIZE && alert_data[i].pcap_id[0]!=0; ++i) {
		trace_data->num_pcaps++;
		if(alert_data[i].confirm == 1) {
			trace_data->num_mali++;
		}
	}

	return 1;
}

/*
 * Preprocessing: Set up vendors' information
 * Components: TP, FP, TN, FN, weight[2]
 */
void pre_vendor_info ()
{
	int i, j;
	
	memset(vendors, 0, sizeof(struct Vendor)*NUMBER_OF_DEVICES);

	for(i=0; i<MAXSIZE && training_data[i].pcap_id[0]!=0; ++i) {
		for(j=0; j<NUMBER_OF_DEVICES; ++j) {
			if(training_data[i].confirm == 1) {
				if(training_data[i].alert[j] == 1) {
					vendors[j].TP++;
				} else {
					vendors[j].FN++;
				}
			} else {
				if(training_data[i].alert[j] == 1) {
					vendors[j].FP++;
				} else {
					vendors[j].TN++;
				}
			}
		}
	}

	for(i=0; i<NUMBER_OF_DEVICES; ++i) {
		if(vendors[i].TP != 0) {
			vendors[i].weight[0] = (float)(training_traces.num_mali*vendors[i].TP)/
			((training_traces.num_mali*vendors[i].TP)+(training_traces.num_pcaps-training_traces.num_mali)*vendors[i].FP);
		}
		if(vendors[i].TN != 0) {
			vendors[i].weight[1] = (float)((training_traces.num_pcaps-training_traces.num_mali)*vendors[i].TN)/
			(((training_traces.num_pcaps-training_traces.num_mali)*vendors[i].TN)+training_traces.num_mali*vendors[i].FN);
		}
		vendors[i].TP = (float)vendors[i].TP / training_traces.num_mali;
        vendors[i].FP = (float)vendors[i].FP / (training_traces.num_pcaps-training_traces.num_mali);
        vendors[i].TN = (float)vendors[i].TN / (training_traces.num_pcaps-training_traces.num_mali);
        vendors[i].FN = (float)vendors[i].FN / training_traces.num_mali;
	}
}

/*
 * Function: selection_sort
 */
void selection_sort (float *elements, int num_elements)
{
	int i, j, index;
	float tmp;

	for(i=0; i<num_elements; ++i) {
		index = i;
		for(j=i+1; j<num_elements; ++j) {
			if(elements[j] < elements[index]) {
				index = j;
			}
		}
		tmp = elements[i];
		elements[i] = elements[index];
		elements[index] = tmp;
	}
}

/*
 * Function: authority_selecting
 * Requirements:	Both FP and FN are in top 25%, ceiling, such as top 2 of 7.
 */
int authority_selecting ()
{
	int i;
	float fp_set[NUMBER_OF_DEVICES], fn_set[NUMBER_OF_DEVICES];
	int top = 0;
	float threshold_authority_fp = 0;
	float threshold_authority_fn = 0;
	float sum_fp, sum_fn; 
	
	top = ceil(NUMBER_OF_DEVICES*0.25);
	//printf("Number of top members == %d\n", top);

	for(i=0; i<NUMBER_OF_DEVICES; ++i) {
		fp_set[i] = vendors[i].FP;
		fn_set[i] = vendors[i].FN;
	}
	selection_sort(fp_set, NUMBER_OF_DEVICES);
	selection_sort(fn_set, NUMBER_OF_DEVICES);

	sum_fp = sum_fn = 0;
	for(i=0; i<top; ++i) {
		sum_fp += fp_set[i];
		sum_fn += fn_set[i];
	}
	threshold_authority_fp = (float) sum_fp / top;
	threshold_authority_fn = (float) sum_fn / top;
	printf("threshold_authority_fp=%f\n", threshold_authority_fp);
	printf("threshold_authority_fn=%f\n", threshold_authority_fn);

	for(i=0; i<NUMBER_OF_DEVICES; ++i) {
		if((vendors[i].FP<=threshold_authority_fp) && (vendors[i].FN<=threshold_authority_fn)) {
			vendors[i].authority = 1;
		}
	}
	
	return 1;
}

/*
 * Function: direct_decision
 * Requirements:	AS: If authority exist, decide the trace directly according to authority detection.
 * 					(VEP: If there is only one voter, decide the trace directly according to the voter.)
 */
void direct_decision()
{
	int i, j;
	int RIA, RNINA;

	RIA = RNINA = 0;
	for(i=0; i<traces.num_pcaps; ++i) {
		for(j=0; j<NUMBER_OF_DEVICES; ++j) {
			if(vendors[j].authority == 0) {
				continue;
			}
			if(data[i].alert[j] == 1) {
				if(data[i].confirm == 1) {
					RIA++;
				}
			} else {
				if(data[i].confirm == 0) {
					RNINA++;
				}
			}
		}
	}
	printf("RIA: %d\n", RIA);
    printf("RNINA: %d\n", RNINA);
    printf("RIA rate: %f\n", (float)RIA/traces.num_mali);
    printf("RNINA rate: %f\n", (float)RNINA/(traces.num_pcaps-traces.num_mali));
}

/*
 * Function: voter_excluding
 * Requirements:	TP rate is less than and FP rate is more than detedction threshold, 0.5.
 * 					Either alert or non-alert rate is more than abnormal threshold, 0.9.
 */
int voter_excluding ()
{
	int i, j;
	int Aflag, NAflag;
	float A, NA;

	for(i=0; i<NUMBER_OF_DEVICES; ++i) {
		if(vendors[i].TP<DETECTION_THRESHOLD && vendors[i].FP>DETECTION_THRESHOLD) {
			vendors[i].weight[0] = 0;
			vendors[i].weight[1] = 0;
			vendors[i].excluded = 1;
		}

		Aflag = NAflag = 0;
		A = NA = 0;
		for(j=0; j<traces.num_pcaps; ++j) {
			if (data[j].alert[i] == 1) {
				Aflag++;
			} else {
				NAflag++;
			}
		}				
		A = (float) Aflag/traces.num_pcaps;
		NA = (float) NAflag/traces.num_pcaps;
		if(A>=ABNORMALITY_THRESHOLD || NA>=ABNORMALITY_THRESHOLD) {
			vendors[i].weight[0] = 0;
			vendors[i].weight[1] = 0;
			vendors[i].excluded = 1;
		}
	}

	return 1;
}

/*
 * Function: weighted_voting
 * Requirements:	Creditability Malicious Decision Function
 * 					Creditability Healthy Decision Function
 *					Two-level: Protocol level, Alert Message level
 */
int weighted_voting (int num_voters, char *db_host, char *db_user, char *db_password, char *db_db)
{
	int i, j;
	int Mflag, Hflag, RIA, RNINA;
	float MCD, HCD;
	int oflag, othersM=0, othersH=0; /* used to check the Unknown traces */
	int FFP, FFN; //addd

	printf("num_voters=%d\n", num_voters);
	/* ------ Frequent FPs Exclusion ------- */
	MYSQL *conn_training;
	conn_training = mysql_init(NULL);
	if(conn_training == NULL) {
		printf("Error %u: %s\n", mysql_errno(conn_training), mysql_error(conn_training));
		exit(1);
	}
	if(mysql_real_connect(conn_training, db_host, db_user, db_password, db_db, 0, NULL, 0) == NULL) {
		printf("Error %u: %s\n", mysql_errno(conn_training), mysql_error(conn_training));
		exit(1);
	}
	/* ------------------------------------- */

	MYSQL_RES *result;
	MYSQL_ROW row;
	char cmd[MAXLINE];

	FFP = FFN = 0; //addd
	RIA = RNINA = 0;
	for(i=0; i<traces.num_pcaps; ++i) {
		Mflag = Hflag = 0;
		MCD = HCD = 0;
		oflag=0;
		printf("====== %d-th trace start ======\n", i);
		for(j=0; j<NUMBER_OF_DEVICES; ++j) {
			if(vendors[j].excluded == 1) {
				continue;
			}
			if(data[i].alert[j] == 1) {	
				memset(cmd, 0, MAXLINE);
		        sprintf(cmd, "SELECT tp, fp, total FROM statistics_msg WHERE msg='%s' AND dev_name='%s'", data[i].msg[j], data[i].dev_name[j]);
		        mysql_query(conn_training, cmd);
		        result = mysql_store_result(conn_training);
				if((row = mysql_fetch_row(result)) != NULL) {
					/* row[0]: tp, row[1]: fp, row[2]: total */
					printf("row[0]=%s\n", row[0]);
					printf("row[1]=%s\n", row[1]);
					printf("row[2]=%s\n", row[2]);
					if(((atol(row[1])/atol(row[2]))==1) && (num_voters==1)) {
						Hflag++;
						continue;
					} else {
						MCD += (atof(row[0])/atof(row[2]));
						HCD += (atof(row[1])/atof(row[2]));
						printf("[Alert Message level] ==Alert==> MCD[%d][%d]=%f\n", i, j, MCD);
						printf("[Alert Message level] ==Alert==> HCD[%d][%d]=%f\n", i, j, HCD);
						continue;
					}
				}
				mysql_free_result(result);
				
				MCD += vendors[j].weight[0];
				HCD += (1-vendors[j].weight[0]);
				printf("[Protocol level] ==Alert==> MCD[%d][%d]=%f\n", i, j, MCD);
				printf("[Protocol level] ==Alert==> HCD[%d][%d]=%f\n", i, j, HCD);
				
			} 
			else if(data[i].alert[j] == 0) {
				MCD += (1-vendors[j].weight[1]);
				HCD += vendors[j].weight[1];
				printf("[Protocol level] ==NoAlert==> MCD[%d][%d]=%f\n", i, j, MCD);
				printf("[Protocol level] ==NoAlert==> HCD[%d][%d]=%f\n", i, j, HCD);
			}
		}
		MCD = MCD / num_voters;
		HCD = HCD / num_voters;
	 	printf("MCD[%d]: %f\n", i, MCD);
        printf("HCD[%d]: %f\n", i, HCD);
		printf("====== %d-th trace finish ======\n", i);

		//if(MCD > HCD) { Mflag++; }
		//else if(MCD < HCD) { Hflag++; }
		if(MCD > ALPHA) { Mflag++; }
		else if(MCD < BETA) { Hflag++; }
		else { oflag++; } /* used to record Unknown cases */

		if(Mflag > 0) {
			data[i].v_result = 1;
			if(data[i].confirm == 1) RIA++;
			else if(data[i].confirm == 0) FFP++; //addd
		}
		if(Hflag > 0) {
			data[i].v_result = 0;
        	if(data[i].confirm == 0) RNINA++;
			else if(data[i].confirm == 1) FFN++; //addd
		}
		/* Unknown traces with actual activities, i.e., Malicious, Benign(Healthy) */
		if(oflag>0) {
			if(data[i].confirm == 1) othersM++;
			if(data[i].confirm == 0) othersH++;
		}
	}
	mysql_close(conn_training);
	printf("RIA: %d\n", RIA);
    printf("RNINA: %d\n", RNINA);
	printf("RIA rate: %f\n", (float)RIA/traces.num_mali);
    printf("RNINA rate: %f\n", (float)RNINA/(traces.num_pcaps-traces.num_mali));
	printf("othersM: %d\n", othersM);
	printf("othersH: %d\n", othersH);
	printf("FFP: %d\n", FFP); //addd
	printf("FFN: %d\n", FFN); //addd

	return 1;
}

/*
 * Function: majority_voting
 */
int majority_voting ()
{
	int i, j;
	int Mflag, Hflag, RIA, RNINA;

	RIA = RNINA = 0;
	for(i=0; i<traces.num_pcaps; ++i) {
		Mflag = Hflag = 0;
		for(j=0; j<NUMBER_OF_DEVICES; ++j) {
			if(data[i].alert[j] == 1) {
				Mflag++;
			} else {
				Hflag++;
			}
		}

		if(Mflag > Hflag) {
			data[i].v_result = 1;
			if(data[i].confirm == 1) RIA++;
		}
		else if(Mflag < Hflag) {
			data[i].v_result = 0;
			if(data[i].confirm == 0) RNINA++;
		}
	}

	printf("RIA: %d\n", RIA);
	printf("RNINA: %d\n", RNINA);
	printf("RIA rate: %f\n", (float)RIA/traces.num_mali);
	printf("RNINA rate: %f\n", (float)RNINA/(traces.num_pcaps-traces.num_mali));

	return 1;
}

/*
 * Function: help_msg()
 */
void help_msg ()
{
	char *msg = "\
Usage: ./cwv [-T Training_data] [-P Processing_data] [-H DB_host] [-u DB_user] [-p DB_password] [-D DB_database]\n\n\
Options:\n\
	-T	training data file (can comprise a directory)\n\
	-P	processing data file (can comprise a directory)(type of trace should be the same as training data)\n\
	-H	database server host\n\
	-u	user name for database connection\n\
	-p	password for database connection\n\
	-D	database name\n\n\
	-m	switch to the majority voting algorithm\n";
	printf ("%s", msg);
}

