#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include <unistd.h>
#include <getopt.h>
#include <search.h>
#include <pcap.h>
#include <signal.h>
#include <errno.h>
#include <pthread.h>
#include <fcntl.h>
#include <err.h>

#include <libnetfilter_queue/libnetfilter_queue.h>
#include <netpacket/packet.h>
#include <arpa/inet.h>

#include <linux/types.h>
#include <linux/netfilter.h>
#include <linux/un.h>

#include <net/ethernet.h>
#include <net/if.h>

#include <netinet/in.h>
#include <netinet/ether.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include "packet_attribute_parser.h"
#include "pcap_dumper.h"
#include "core.h"
#include "hash_map.h"
#include "main.h"
#include "blacklist_module.h"
#include "stateful_module.h"
#include "decoy.h"



/* global variables */
struct dispatcher_config config;
int save_memory = 0;
static struct option long_options[] = {
	/* required */
	{"bot-mac", required_argument, NULL, 'm'},
	{"gateway-mac", required_argument, NULL, 'M'},
	{"verdict", required_argument, NULL, 'v'},


	/* optional: three phases of traffic retargeting */
	{"bot-interface", required_argument, NULL, 'i'},
	{"bot-ip", required_argument, NULL, 'I'},
	{"decoy-interface", required_argument, NULL, 'd'},
	{"decoys", required_argument, NULL, 'D'},
	{"snort-usock", required_argument, NULL, 'j'},	/* -j the same as my_snort */
	/* deprecated
	{"emulator-interface", required_argument, NULL, 'e'},
	{"emulator-controller-ip", required_argument, NULL, 'E'},
	*/

	/* optional: miscellanceous */
	{"black-list", required_argument, NULL, 'b'},
	{"log", required_argument, NULL, 'l'},

	{"queue-num", required_argument, NULL, 'Q'},
	/* deprecated
	{"reinject", required_argument, NULL, 'r'},
	{"queued-packets-pcap-dump-filename", required_argument, NULL, 'x'},
	{"emulation-packets-pcap-dump-filename", required_argument, NULL, 'y'},
	*/

	{"quiet", no_argument, NULL, 'q'},
	{"help", no_argument, NULL, 'h'},
	{"version", no_argument, NULL, 'V'},
	{0, 0, 0, 0}
};



/* function prototype */
int dispatcher_main();
int dispatcher_callback(struct nfq_q_handle * qh, struct nfgenmsg * nfmsg, struct nfq_data * nfa, void * data);

int rebuild_packet(struct nfq_data * nfa, void * data, char * packet);

/* deprecated */
void reinject_callback(u_char * useless, const struct pcap_pkthdr * pkthdr, const u_char * packet);


void * pthread_start_ipc_snort(void * arg);

void print_statistic();
void hook_atexit();
void signal_handler(int sig);

void version();
void help();



/*
 * Purpose: program starting point, parse argument, set configuration
 * Arguments: command line argument
 * Returns: 0 => normal exit, 1 => exit on error
 */
int main(int argc, char ** argv)
{
	char ch;

	memset(&config, 0, sizeof(struct dispatcher_config));

	/* parse arguments */
	while((ch = getopt_long(argc, argv, "m:M:v:i:I:e:E:d:D:j:b:l:Q:r:x:y:qhV", long_options, NULL)) != -1) {
		switch(ch) {
			/* required */
			case 'm':
				config.bot_mac_on = 1;
				memcpy(&config.bot_mac, (char *) ether_aton(optarg), sizeof(struct ether_addr));
				break;
			case 'M':
				config.gateway_mac_on = 1;
				memcpy(&config.gateway_mac, (char *) ether_aton(optarg), sizeof(struct ether_addr));
				break;
			case 'v':	/* ACCEPT or DROP */
				config.verdict_on = 1;
				if(optarg[0] == 'A' || optarg[0] == 'a') config.verdict = NF_ACCEPT;
				else config.verdict = NF_DROP;
				break;


			/* optional: three phase of change traffic path */
			case 'i':
				config.bot_if_on = 1;
				strncpy(config.bot_if, optarg, IF_LENGTH - 1);
				break;
			case 'I':
				config.bot_ip_on = 1;
				strncpy(config.bot_ip, optarg, IP_LENGTH - 1);
				break;
			/* deprecated
			case 'e':
				config.emulator_if_on = 1;
				strncpy(config.emulator_if, optarg, IF_LENGTH - 1);
				break;
			case 'E':
				config.ipc_emulator_controller_on = 1;
				{
					char * p;

					if((p = index(optarg, ':')) == NULL) { config.ipc_emulator_controller_port = IPC_EMULATOR_CONTROLLER_DEFAULT_PORT; }
					else { config.ipc_emulator_controller_port = strtol(p+1, NULL, 10); *p = 0; }
					strncpy(config.ipc_emulator_controller_ip, optarg, IP_LENGTH - 1);
				}
				break;
			*/
			case 'd':
				config.decoy_if_on = 1;
				strncpy(config.decoy_if, optarg, IF_LENGTH - 1);
				break;
			case 'D':
				config.decoy_controller_on = 1;
				if(!decoy_init(optarg)) { errx(1, "decoy_init"); }
				if(!stateful_module_init()) { errx(1, "stateful_module_init"); }
				break;
			case 'j':
				config.ipc_snort_on = 1;
				strncpy(config.ipc_snort_path, optarg, UNIX_PATH_MAX - 1);
				break;
			
			/* optional: miscellaneout */
			case 'b':
				config.blacklist_on = 1;
				if(!blacklist_module_init(optarg)) { errx(1, "blacklist_module_init"); }
				break;
			case 'l':
				config.log_on = 1;
				strncpy(config.log_filename, optarg, FILENAME_LENGTH - 1);
				break;

			case 'Q': config.queue_number = strtol(optarg, NULL, 10); break;
			/* deprecated
			case 'r':
				config.reinject_on = 1;
				strncpy(config.reinject_filename, optarg, FILENAME_LENGTH - 1);
				break;
			case 'x':
				config.queued_packets_pcap_dump_on = 1;
				strncpy(config.queued_packets_pcap_dumper.pcap_dump_filename, optarg, FILENAME_LENGTH - 1);
				break;
			case 'y':
				config.emulation_packets_pcap_dump_on = 1;
				strncpy(config.emulation_packets_pcap_dumper.pcap_dump_filename, optarg, FILENAME_LENGTH - 1);
				break;
			*/

			case 'q': config.quiet = 1; break;
			case 'h': help(); break;
			case 'V': version(); break;
			default: help(); break;
		}
	}

	if(!config.bot_mac_on || !config.gateway_mac_on || !config.verdict_on) {
		errx(1, "required -m -M -v arguments");
	}
	if((config.bot_if_on | config.bot_ip_on | config.decoy_if_on | config.decoy_controller_on | config.ipc_snort_on) &&
	  !(config.bot_if_on & config.bot_ip_on & config.decoy_if_on & config.decoy_controller_on & config.ipc_snort_on)) {
		errx(1, "options -i -I -e -E -j are binded");
	}

	atexit(hook_atexit);

	return dispatcher_main();
}

/*
 * Purpose: dispatcher main code, setup environment according to the config
 * Arguments: none
 * Returns: 0 => normal exit, 1 => exit on error
 */
int dispatcher_main()
{
	struct nfq_handle * h;
	struct nfq_q_handle * qh;

	/* log should init first */
	if(config.log_on) { if((config.log_fd = open(config.log_filename, O_WRONLY|O_CREAT|O_TRUNC, 0644)) == -1) { err(1, "dispatcher_main: open"); } }

	/* netfilter_queue start up */
	{
		if((h = nfq_open()) == NULL) { errx(1, "dispatcher_main: nfq_open"); }
		if(nfq_unbind_pf(h, AF_INET) != 0) { errx(1, "dispatcher_main: nfq_unbind_pf"); }
		if(nfq_bind_pf(h, AF_INET) != 0) { errx(1, "dispatcher_main: nfq_bind_pf"); }
		if((qh = nfq_create_queue(h, config.queue_number, &dispatcher_callback, NULL)) == 0) { errx(1, "dispatcher_main: nfq_create_queue"); }
		if(nfq_set_mode(qh, NFQNL_COPY_PACKET, 0xffff) == -1) { errx(1, "dispatcher_main: nfq_set_mode"); }

		warnx("bind at queue %d", config.queue_number);
	}

	{
		if(hash_map_init() == -1) { errx(1, "dispatcher_main: hash_map_init"); }
	}

	/* IPC with snort start up */
	if(config.ipc_snort_on) {
		struct sockaddr_un saddr;
		pthread_t thread;

		if((config.ipc_snort_fd = socket(AF_UNIX, SOCK_DGRAM, 0)) == -1) { err(1, "dispatcher_main: socket"); }

		memset(&saddr, 0, sizeof(struct sockaddr_un));
		saddr.sun_family = PF_UNIX;
		strncpy(saddr.sun_path, config.ipc_snort_path, sizeof(saddr.sun_path) - 1);

		if(bind(config.ipc_snort_fd, (struct sockaddr *) &saddr, sizeof(struct sockaddr_un)) == -1) { err(1, "dispatcher_main: bind"); }

		if(pthread_create(&thread, NULL, pthread_start_ipc_snort, NULL) != 0) { err(1, "dispatcher_main: pthread_create"); }

		warnx("IPC with snort over datagram");
	}

	/* IPC with emulator_controller start up */
#if 0
	if(config.ipc_emulator_controller_on) {
		if((config.ipc_emulator_controller_fd = socket(AF_INET, SOCK_STREAM, 0)) == -1) { err(1, "dispatcher_main: socket"); }

		memset(&config.ipc_emulator_controller_addr, 0, sizeof(struct sockaddr_in));
		config.ipc_emulator_controller_addr.sin_family = AF_INET;
		if(inet_pton(AF_INET, config.ipc_emulator_controller_ip, (void *) &config.ipc_emulator_controller_addr.sin_addr) < 0) { err(1, "dispatcher_main: inet_pton"); }
		config.ipc_emulator_controller_addr.sin_port = htons(config.ipc_emulator_controller_port);

		if(connect(config.ipc_emulator_controller_fd, (struct sockaddr *) &config.ipc_emulator_controller_addr, sizeof(struct sockaddr_in)) == -1) {
			warnx("dispatcher_main: connect : %s:%d", config.ipc_emulator_controller_ip, config.ipc_emulator_controller_port);
			err(1, "dispatcher_main: connect : %s:%d", config.ipc_emulator_controller_ip, config.ipc_emulator_controller_port);
		}

		if(config.emulation_packets_pcap_dump_on) { if(!pcap_dumper_init(&config.emulation_packets_pcap_dumper)) { errx(1, "dispatcher_main: pcap_dumper_init"); } }

		warnx("IPC with emulator controller over tcp");
	}
#endif

	/* redirection */
	if(config.decoy_if_on) {
		struct ifreq req;
		int fd;

		/* dummy socket */
		if((fd = socket(PF_PACKET, SOCK_RAW, 0)) == -1) { err(1, "dispatcher_main: socket"); }

		/* get ifindex */
		memset(&req, 0, sizeof(struct ifreq));
		strncpy(req.ifr_name, config.decoy_if, sizeof(req.ifr_name) - 1);
		req.ifr_name[sizeof(req.ifr_name) - 1] = 0;
		if(ioctl(fd, SIOCGIFINDEX, &req) == -1) { err(1, "dispatcher_main: ioctl"); }
		config.decoy_if_index = req.ifr_ifindex;

		/* open promiscous mode */
		memset(&req, 0, sizeof(struct ifreq));
		strncpy(req.ifr_name, config.decoy_if, sizeof(req.ifr_name) - 1);
		req.ifr_name[sizeof(req.ifr_name) - 1] = 0;
		if(ioctl(fd, SIOCGIFFLAGS, &req) == -1) { err(1, "dispatcher_main: ioctl"); }
		req.ifr_flags |= IFF_PROMISC;
		if(ioctl(fd, SIOCSIFFLAGS, &req) == -1) { err(1, "dispatcher_main: ioctl"); }
		close(fd); /* we don't care if close error */
	}

	/* redirection write back mechanism */
	if(config.bot_if_on) {
		struct ifreq req;
		int fd;

		/* get ifindex */
		memset(&req, 0, sizeof(struct ifreq));
		strncpy(req.ifr_name, config.bot_if, sizeof(req.ifr_name) - 1);
		if((fd = socket(PF_PACKET, SOCK_RAW, 0)) == -1) { err(1, "dispatcher_main: socket"); }
		if(ioctl(fd, SIOCGIFINDEX, &req) == -1) { err(1, "dispatcher_main: ioctl"); }
		close(fd); /* we don't care if close error */

		config.bot_if_index = req.ifr_ifindex;
	}

	/* deprecated
	if(config.queued_packets_pcap_dump_on) { if(!pcap_dumper_init(&config.queued_packets_pcap_dumper)) { errx(1, "dispatcher_main: pcap_dumper_init"); } }
	*/

	if(config.reinject_on) {
		pcap_t * pcap;
		char errbuf[PCAP_ERRBUF_SIZE];

		if((pcap = pcap_open_offline(config.reinject_filename, errbuf)) == NULL) errx(1, "dispatcher_main: pcap_open_offline");
		pcap_loop(pcap, -1, reinject_callback, NULL);
	}

	/* signal register */
	{
		if(signal(SIGINT, signal_handler) == SIG_ERR) { err(1, "dispatcher_main: signal"); }
		if(signal(SIGTERM, signal_handler) == SIG_ERR) { err(1, "dispatcher_main: signal"); }
	}

	/* main part */
	{
		char buf[BUFSIZ];
		int fd, rv;

		/* no error return value */
		fd = nfq_fd(h);

		while((rv = recv(fd, buf, BUFSIZ, 0)) >= 0) {
			if(nfq_handle_packet(h, buf, rv) != 0) {
				errx(1, "dispatcher_main: nfq_handle_packet");
			}
		}
	}

	/* actually, never reach here */
	abort();

	return 0;
}

/*
 * Purpose: callback function for netfilter_queue
 * Arguments: specify by netfilter_queue API
 * Returns: specify by netfilter_queue API
 */
int dispatcher_callback(struct nfq_q_handle * qh, struct nfgenmsg * nfmsg, struct nfq_data * nfa, void * data)
{
	struct nfqnl_msg_packet_hdr * ph;
	char packet[BUFSIZ];
	int packet_len;
	struct packet_attribute attr;

	if((ph = nfq_get_msg_packet_hdr(nfa)) == 0) { errx(1, "dispatcher_callback: nfq_get_msg_packet_hdr"); }
	
	memset(packet, 0, BUFSIZ);
	if((packet_len = rebuild_packet(nfa, data, packet)) == -1) { warnx("dispatcher_callback: rebuild_packet"); goto UNKNOWN; }
	attr = packet_parse(packet);

	/* statistic */
	config.packet_count++;
	config.byte_count += packet_len;
	if(!config.replay_timestamp_on) {
		config.replay_timestamp_on = 1;
		if(gettimeofday(&config.replay_timestamp, NULL) == -1) { warn("dispatcher_callback: gettimeofday"); }
	}

	if(attr.ip) {
		char sip[IP_LENGTH], dip[IP_LENGTH];
		struct session * session;
		struct packet * ppacket = NULL;

		if(inet_ntop(AF_INET, &attr.ip_header->saddr, sip, IP_LENGTH) == NULL) { err(1, "dispatcher_callback: inet_ntop: format error"); }
		if(inet_ntop(AF_INET, &attr.ip_header->daddr, dip, IP_LENGTH) == NULL) { err(1, "dispatcher_callback: inet_ntop: format error"); }
		if((session = get_session(sip, dip)) == NULL) { errx(1, "dispatcher_callback: get_session"); }

		if(attr.tcp) {
			struct connection * connection;

			if((connection = get_connection(session, ntohs(attr.tcp_header->source), ntohs(attr.tcp_header->dest))) == NULL) { errx(1, "dispatcher_callback: get_connection"); }

			/* statistic */
			pthread_mutex_lock(&session->mutex_obj);
			if(session->status == SESSION_PATH_CHANGED_STATUS) {
				switch(connection->status) {
					case CONNECTION_REDIRECT_STATUS:
						config.redirected_packet_count++;
						config.redirected_byte_count += packet_len;
						break;
					case CONNECTION_RELAY_STATUS:
						config.relayed_packet_count++;
						config.relayed_byte_count += packet_len;
						break;
					default:
						if(!save_memory) warnx("no, impossible area");
						break;
				}

				config.drop_packet_count++;
				config.drop_byte_count += packet_len;
			}
			pthread_mutex_unlock(&session->mutex_obj);

			if(config.decoy_if_on) { 
				if((ppacket = get_packet(connection, packet, packet_len)) == NULL) { errx(1, "dispatcher_callback: get_packet"); }
			}
		}

		if(ppacket != NULL) return nfq_set_verdict(qh, ntohl(ph->packet_id), get_verdict(ppacket), 0, NULL);
		else goto UNKNOWN;
	}

UNKNOWN:
	/* default verdict */
	return nfq_set_verdict(qh, ntohl(ph->packet_id), config.verdict, 0, NULL);
}

/* Purpose: rebuild packet buffer
 * Arguments:
 * Returns: none
 * Bug: 1. if in the same subnet
 *      2. byte order in protocol
 */
int rebuild_packet(struct nfq_data * nfa, void * data, char * packet)
{
    int packet_len;

    packet_len = 0;

    /* get source mac address */
    {
        int i;
        struct nfqnl_msg_packet_hw * hwph;

        if((hwph = nfq_get_packet_hw(nfa)) < 0) { warnx("nfq_get_packet_hw"); return -1; }

        if(memcmp(&config.bot_mac, hwph->hw_addr, sizeof(struct ether_addr)) == 0) {
            /* outgoing packet */
            for(i=0; i<sizeof(struct ether_addr); ++i)
                packet[packet_len++] = config.gateway_mac.ether_addr_octet[i];
            for(i=0; i<sizeof(struct ether_addr); ++i)
                packet[packet_len++] = config.bot_mac.ether_addr_octet[i];
        } else {
            /* incoming packet */
            for(i=0; i<sizeof(struct ether_addr); ++i)
                packet[packet_len++] = config.bot_mac.ether_addr_octet[i];
            for(i=0; i<sizeof(struct ether_addr); ++i)
                packet[packet_len++] = config.gateway_mac.ether_addr_octet[i];
        }
    }

    /* get protocol id */
    {
        char * p;
        struct nfqnl_msg_packet_hdr * ph;
        u_int16_t protocol;

        if((ph = nfq_get_msg_packet_hdr(nfa)) < 0) { warnx("nfq_get_msg_packet_hdr"); return -1; }

        /* byte order depend on OS? */
        //protocol = ntohs(ph->hw_protocol);
        protocol = ph->hw_protocol;
        p = (char *) &protocol;
        packet[packet_len++] = p[0];
        packet[packet_len++] = p[1];
    }

    /* get layer2 payload */
    {
        char * p;
        int len, i;

        p = (char *) data;
        if((len = nfq_get_payload(nfa, (unsigned char **) &p)) < 0) { warnx("nfq_get_payload"); return -1; }
        for(i=0; i<len; ++i) packet[packet_len++] = p[i];
    }

    return packet_len + 1;
}

/*
 * Purpose: reinject pcap into packet queue
 * Arguments: libpcap callback function
 * Returns: none
 * Comment: deprecated
 */
void reinject_callback(u_char * useless, const struct pcap_pkthdr * pkthdr, const u_char * packet)
{
#if 0
	struct packet_attribute attr;

	attr = packet_parse((char *) packet);

	if(attr.ip) {
		char sip[IP_LENGTH], dip[IP_LENGTH];
		struct session * session;

		memset(sip, 0, IP_LENGTH);
		memset(dip, 0, IP_LENGTH);
		if(inet_ntop(AF_INET, &attr.ip_header->saddr, sip, IP_LENGTH) == NULL) { err(1, "reinject_callback: inet_ntop"); }
		if(inet_ntop(AF_INET, &attr.ip_header->daddr, dip, IP_LENGTH) == NULL) { err(1, "reinject_callback: inet_ntop"); }
		session = libdispatcher_get_session(sip, dip);

		if(attr.tcp) {
			struct connection * connection;

			connection = libdispatcher_get_connection(session, ntohs(attr.tcp_header->source), ntohs(attr.tcp_header->dest));

			if(config.emulator_if_on) {
				libdispatcher_queue_packet(session, connection, (char *) packet, (int) pkthdr->len);
			}
		}
	}
#endif
}

/*
 * Purpose: pthread start routine for ipc with snort
 * Arguments: none
 * Returns: none
 */
void * pthread_start_ipc_snort(void * arg)
{
	char sip[IP_LENGTH], dip[IP_LENGTH];
	char alert_message[BUFSIZ];
	char buf[BUFSIZ];
	struct session * session;
	int ret;

	/* read alert from snort forever */
	while(1) {
		memset(buf, 0, BUFSIZ);
		if((ret = recvfrom(config.ipc_snort_fd, buf, BUFSIZ - 1, 0, NULL, NULL)) == -1) { err(1, "pthread_start_ipc_snort: recvfrom"); }
		else if(ret == 0) { errx(1, "ipc snort recvfrom end"); break; }

		/* parse messsage */
		{
			char * p, * tmp_p;

			if((p = strtok_r(buf, ":", &tmp_p)) == NULL) { errx(1, "pthread_start_ipc_snort: strtok_r"); } memset(sip, 0, IP_LENGTH); strncpy(sip, p, IP_LENGTH - 1);
			if((p = strtok_r(NULL, ":", &tmp_p)) == NULL) { errx(1, "pthread_start_ipc_snort: strtok_r"); } memset(dip, 0, IP_LENGTH); strncpy(dip, p, IP_LENGTH - 1);
			if((p = strtok_r(NULL, ":", &tmp_p)) == NULL) { errx(1, "pthread_start_ipc_snort: strtok_r"); } memset(alert_message, 0, BUFSIZ); strncpy(alert_message, p, BUFSIZ - 1);

			if(strcmp(config.bot_ip, dip) == 0) {
				char tip[IP_LENGTH];

				/* swap sip and dip */
				memset(tip, 0, IP_LENGTH); strncpy(tip, dip, IP_LENGTH - 1);
				memset(dip, 0, IP_LENGTH); strncpy(dip, sip, IP_LENGTH - 1);
				memset(sip, 0, IP_LENGTH); strncpy(sip, tip, IP_LENGTH - 1);
			}
		}

		/* statistic */
		config.alert_message_count++;

		{
			struct timeval gg;

			memset(&gg, 0, sizeof(struct timeval));
			gettimeofday(&gg, NULL);
			warnx("%ld.%ld snort alert [%s] for session %s_%s", gg.tv_sec, gg.tv_usec, alert_message, sip, dip);
		}

		if((session = get_session(sip, dip)) == NULL) { warnx("no such session %s_%s", sip, dip); }
		else {
			/* TODO: should set STATUS_ALERTED */
			if(!config.blacklist_on) start_change_path(session);
		}
	}

	/* actually, un-reachable */
	warnx("pthread_start_ipc_snort die");

	pthread_exit(0);
	return NULL;
}

/*
 * Purpose: print statistic information
 * Arguments: none
 * Returns: none
 */
void print_statistic()
{
	/* under 64 bits system, use ld instead of lld */
#ifdef __i386__
#define __P__ "%lld"
#endif
#ifdef __x86_64__
#define __P__ "%ld"
#endif
	warnx("byte_count="__P__"", config.byte_count);
	warnx("packet_count="__P__"", config.packet_count);
	warnx("connection_count="__P__"", config.connection_count);
	warnx("session_count="__P__"", config.session_count);

	warnx("replay_session_count="__P__"", config.replay_session_count);

	warnx("replay_timestamp=%ld.%ld", config.replay_timestamp.tv_sec, config.replay_timestamp.tv_usec);
	warnx("replayed_byte_count="__P__"", config.replayed_byte_count);
	warnx("replayed_packet_count="__P__"", config.replayed_packet_count);
	warnx("replayed_connection_count="__P__"", config.replayed_connection_count);

	warnx("redirect_timestamp=%ld.%ld", config.redirect_timestamp.tv_sec, config.redirect_timestamp.tv_usec);
	warnx("redirected_byte_count="__P__"", config.redirected_byte_count);
	warnx("redirected_packet_count="__P__"", config.redirected_packet_count);
	warnx("redirected_connection_count="__P__"", config.redirected_connection_count);

	warnx("relay_timestamp=%ld.%ld", config.relay_timestamp.tv_sec, config.relay_timestamp.tv_usec);
	warnx("relayed_byte_count="__P__"", config.relayed_byte_count);
	warnx("relayed_packet_count="__P__"", config.relayed_packet_count);
	warnx("relayed_connection_count="__P__"", config.relayed_connection_count);

	warnx("drop_byte_count="__P__"", config.drop_byte_count);
	warnx("drop_packet_count="__P__"", config.drop_packet_count);
	warnx("drop_connection_count="__P__"", config.drop_connection_count);
	warnx("drop_session_count="__P__"", config.drop_session_count);

	warnx("alert_message_count="__P__"", config.alert_message_count);
}

void hook_atexit()
{
	if(config.ipc_snort_on) unlink(config.ipc_snort_path);
	print_statistic();
}

/*
 * Purpose: signal handler
 * Arguments: signal number
 * Returns: none
 */
void signal_handler(int sig)
{
	switch(sig) {
		case SIGINT:
		case SIGTERM:
			exit(0);
			break;
		default: warnx("strange signal %d", sig); break;
	}
}

/*
 * Purpose: print version message
 * Arguments: none
 * Returns: none
 */
void version()
{
	printf("%s\n", "Dispatcher 1.0");

	exit(0);
}

/*
 * Purpose: print help message
 * Arguments: none
 * Returns: none
 */
void help()
{
	printf("%s\n", "This is an implementation of `Secure and Transparent Network Traffic Replay, Redirect and Relay in a Dynamic Malware Analysis Environment` thesis.");

	printf("%s\n", "Usage: dispatcher -m bot-mac -M gateway-mac -v verdict [OPTION]...");

	printf("%s\n", "Options:");
	/* required */
	printf("%-40s\n", "[Required]");
	printf(" %-40s%-40s\n", "-m, --bot-mac=MAC", "bot's mac address");
	printf(" %-40s%-40s\n", "-M, --gateway-mac=MAC", "gateway's mac address");
	printf(" %-40s%-40s\n", "-v, --verdict=STRING", "default verdict, either 'ACCEPT' or 'DROP'");

	/* optional: three phases of traffic retargeting */
	printf("%-40s\n", "[Optional: three phases of traffic retargeting]");
	printf(" %-40s%-40s\n", "-i, --bot-interface=IF", "bot-side NIC name");
	printf(" %-40s%-40s\n", "-I, --bot-ip=IP", "bot's IP address");
	printf(" %-40s%-40s\n", "-d, --decoy-interface=IF", "decoy-side NIC name");
	printf(" %-40s%-40s\n", "-D, --decoys=IP/MAC/flag", "decoys' IP, MAC addresses and flag (seperated by ',', details in EXAMPLES)");
	printf(" %-40s%-40s\n", "-j, --snort-usock=PATH", "UNIX domain socket path from Snort");
	printf(" %-40s%-40s\n", "-e, --emulator-interface=IF", "emulator-side NIC name (deprecated)");
	printf(" %-40s%-40s\n", "-E, --emulator-controller-ip=IP", "emulator controller's IP address, specify port by ':' (deprecated)");

	/* optional: miscellaneous */
	printf("%-40s\n", "[Optional: miscellaneous]");
	printf(" %-40s%-40s\n", "-b, --black-list=[IP:port]", "black list (seperated by ',', details in EXAMPLES)");
	printf(" %-40s%-40s\n", "-l, --log=FILE", "log");

	printf(" %-40s%-40s\n", "-Q, --queue-num=NUM", "netfilter queue number");
	printf(" %-40s%-40s\n", "-q, --quiet", "quiet mode");
	printf(" %-40s%-40s\n", "-h, --help", "show this message");
	printf(" %-40s%-40s\n", "-V, --version", "show version message");

	printf(" %-40s%-40s\n", "-r, --reinject=FILE", "reinject pcap file (deprecated)");
	printf(" %-50s%-40s\n", "-x, --queued-packets-pcap-dump-filename=FILE", "dump pcap (deprecated)");
	printf(" %-50s%-40s\n", "-y, --emulation-packets-pcap-dump-filename=FILE", "dump pcap (deprecated)");

	exit(0);
}

