#include "packet_attribute_parser.h"
#include "hash_map.h"
#include "main.h"
#include "err.h"
#include "core.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <arpa/inet.h>

struct layer4_state_bucket * get_layer4_state_bucket(struct connection * connection);

struct layer4_state_bucket {
#ifndef STATE_BUCKET_KEY_LENGTH
#define STATE_BUCKET_KEY_LENGTH 100
#endif
	char key[STATE_BUCKET_KEY_LENGTH];

#ifndef TCP_STATE_NULL
#define TCP_STATE_NULL 0
#endif
#ifndef TCP_STATE_SYN_SENT
#define TCP_STATE_SYN_SENT 1
#endif
#ifndef TCP_STATE_SYN_ACKED
#define TCP_STATE_SYN_ACKED 2
#endif
#ifndef TCP_STATE_ESTABLISHED
#define TCP_STATE_ESTABLISHED 3
#endif
#ifndef TCP_STATE_FIN_SENT
#define TCP_STATE_FIN_SENT 4
#endif
#ifndef TCP_STATE_LAST_ACK
#define TCP_STATE_LAST_ACK 5
#endif
#ifndef TCP_STATE_DEAD
#define TCP_STATE_DEAD 100
#endif
    u_int32_t tcp_state;
    u_int32_t tcp_b_ack;    /* write to bot's ack */
    u_int32_t tcp_b_seq;
    u_int32_t tcp_e_ack;    /* write to emulator's ack */
};

void * default_layer4_write_module(struct connection * connection, struct packet_attribute attr)
{
	struct layer4_state_bucket * bucket;

	if(!attr.ip || !attr.tcp) return NULL;

	if((bucket = get_layer4_state_bucket(connection)) == NULL) { warnx("default_layer4_write_module: hash_map_get"); return NULL; }

	/* the highest priority flag */
	if(attr.tcp_header->rst) {
		bucket->tcp_state = TCP_STATE_DEAD;
		if(attr.tcp_header->ack) { attr.tcp_header->ack_seq = bucket->tcp_e_ack; }
	} else {
		switch(bucket->tcp_state) {
			case TCP_STATE_NULL:
				if(attr.tcp_header->syn) {
					bucket->tcp_state = TCP_STATE_SYN_SENT;
					bucket->tcp_b_ack = htonl(ntohl(attr.tcp_header->seq) + attr.tcp_data_len + 1);
				} else { warnx("impossible"); return NULL; }
				break;
			case TCP_STATE_SYN_ACKED:
				if(attr.tcp_header->ack) {
					bucket->tcp_state = TCP_STATE_ESTABLISHED;
					bucket->tcp_b_ack = htonl(ntohl(attr.tcp_header->seq) + attr.tcp_data_len + 1);
					//bucket->tcp_b_ack = htonl(ntohl(attr.tcp_header->seq) + attr.tcp_data_len); /* in smtp no need +1 */
					bucket->tcp_b_seq = attr.tcp_header->ack_seq;

					attr.tcp_header->ack_seq = bucket->tcp_e_ack;

					/* in smtp no need this */
					//                              if(attr.tcp_data_len == 0) write_again_flag = 1;    /* three way handshake last ack */
				} else { warnx("impossible"); return NULL; }
				break;
			case TCP_STATE_ESTABLISHED:
				//                          if(ntohl(attr.tcp_header->seq) < bucket->tcp_b_ack) { libdispatcher_warnx("[replay %s] retransmission? Trigger selective replay.", bucket->key); libdispatcher_warnx("send_seq=%u b_ack=%u", ntohl(attr.tcp_header->seq), bucket->tcp_b_ack); goto free_packet; }
				if(attr.tcp_header->fin) {
					bucket->tcp_state = TCP_STATE_FIN_SENT;
				} else {
					bucket->tcp_b_ack = htonl(ntohl(attr.tcp_header->seq) + attr.tcp_data_len);
					//				if(attr.tcp_data_len == 0) write_again_flag = 1;
				}
				if(attr.tcp_header->ack) {
					bucket->tcp_b_seq = attr.tcp_header->ack_seq;
					attr.tcp_header->ack_seq = bucket->tcp_e_ack;
				}
				break;
			case TCP_STATE_LAST_ACK:
				if(attr.tcp_header->ack) {
					bucket->tcp_b_seq = attr.tcp_header->ack_seq;
					attr.tcp_header->ack_seq = bucket->tcp_e_ack;
					//				write_again_flag = 1;
				} else { warnx("impossible"); return NULL; }
				break;

			case TCP_STATE_DEAD: return NULL; break;
			default: warnx("impossible"); return NULL; break;
		}
	}

	return NULL;
}

void * default_layer4_read_module(struct connection * connection, struct packet_attribute attr)
{
	struct layer4_state_bucket * bucket;

	if(!attr.ip || !attr.tcp) return NULL;

	if((bucket = get_layer4_state_bucket(connection)) == NULL) { warnx("default_layer4_read_module: get_layer4_state_bucket"); return NULL; }

	if(attr.tcp_header->rst) {
		bucket->tcp_state = TCP_STATE_DEAD;
	} else {
		switch(bucket->tcp_state) {
			case TCP_STATE_SYN_SENT:
				if(attr.tcp_header->ack) {
					bucket->tcp_state = TCP_STATE_SYN_ACKED;
					bucket->tcp_e_ack = htonl(ntohl(attr.tcp_header->seq) + attr.tcp_data_len + 1);
				} else { warnx("impossible"); return NULL; }
				break;
			case TCP_STATE_ESTABLISHED:
				bucket->tcp_e_ack = htonl(ntohl(attr.tcp_header->seq) + attr.tcp_data_len);
				/* we ignore fin flag */
				break;
			case TCP_STATE_FIN_SENT:
				if(attr.tcp_header->fin && attr.tcp_header->ack) {
					bucket->tcp_state = TCP_STATE_LAST_ACK;
					bucket->tcp_e_ack = htonl(ntohl(attr.tcp_header->seq) + attr.tcp_data_len + 1);
				} else { warnx("impossible"); return NULL; }
				break;

			case TCP_STATE_DEAD: break;
			default: warnx("impossible"); break;
		}
	}

	return NULL;
}

void * default_layer4_writeback_module(struct connection * connection, struct packet_attribute attr)
{
	struct layer4_state_bucket * bucket;

	if(!attr.ip || !attr.tcp) return NULL;

	if((bucket = get_layer4_state_bucket(connection)) == NULL) { warnx("default_layer4_writeback_module: get_layer4_state_bucket"); return NULL; }

	if(bucket->tcp_state != TCP_STATE_SYN_ACKED) {
		attr.tcp_header->seq = bucket->tcp_b_seq;
	}
	if(attr.tcp_header->ack) {
		attr.tcp_header->ack_seq = bucket->tcp_b_ack;
	}

	return NULL;
}

struct layer4_state_bucket * get_layer4_state_bucket(struct connection * connection)
{
	char key[STATE_BUCKET_KEY_LENGTH];
	struct layer4_state_bucket * bucket;

	sprintf(key, "l4_%s:%d_%s:%d", connection->session->sip, connection->sport, connection->session->dip, connection->dport);

	if((bucket = hash_map_get(key)) == NULL) {
		if((bucket = (struct layer4_state_bucket *) malloc(sizeof(struct layer4_state_bucket))) == NULL) { warn("get_layer4_state_bucket: malloc"); return NULL; }
		memset(bucket, 0, sizeof(struct layer4_state_bucket));
		strncpy(bucket->key, key, STATE_BUCKET_KEY_LENGTH - 1);
		if(hash_map_put(key, bucket) == -1) { warnx("get_layer4_state_bucket: hash_map_put"); return NULL; }
	}

	return bucket;
}

