#include "main.h"
#include "core.h"
#include "packet_attribute_parser.h"
#include "hash_map.h"
#include "err.h"
#include "blacklist_module.h"
#include "decoy.h"
#include "stateful_module.h"
#include "pcap_dumper.h"

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <errno.h>

#include <sys/time.h>

extern int save_memory;
extern struct dispatcher_config config;

void remove_connection(struct session * session, struct connection * connection);

/*
 * Purpose: get the session by given session_key, new if needed
 * Arguments: session key
 * Returns: the target session
 */
struct session * get_session(char * sip, char * dip)
{
	char session_key[SESSION_KEY_LENGTH];
	struct session * session;

	memset(session_key, 0, SESSION_KEY_LENGTH);
	sprintf(session_key, "%s_%s", sip, dip);

	/* try to find the corresponding session */
	if((session = hash_map_get(session_key)) == NULL) {
		/* init a new session */
		if((session = (struct session *) malloc(sizeof(struct session))) == NULL) { warn("get_session: malloc"); return NULL; }
		memset(session, 0, sizeof(struct session));
		if(pthread_mutex_init(&session->mutex_obj, NULL) != 0) { warnx("get_session: pthread_mutex_init"); return NULL; }
		if(pthread_cond_init(&session->cond_obj, NULL) != 0) { warnx("get_session: pthread_cont_init"); return NULL; }
		strncpy(session->key, session_key, SESSION_KEY_LENGTH - 1);
		strncpy(session->sip, sip, IP_LENGTH - 1);
		strncpy(session->dip, dip, IP_LENGTH - 1);

		if(hash_map_put(session_key, session) == -1) { warnx("get_session: map_put"); return NULL; }

		warnx("A new session [%s]", session_key);

		/* statistic */
		config.session_count++;

		if(config.blacklist_on && blacklist_module(dip, 0)) { start_change_path(session); }
	}

	return session;
}

void remove_connection(struct session * session, struct connection * connection)
{
	struct connection * c;

	pthread_mutex_lock(&session->mutex_obj);
	if(session->connections == connection) session->connections = session->connections->next;
	else {
		for(c=session->connections; c->next!=NULL; c=c->next) {
			if(c->next == connection) {
				c->next = connection->next;
				break;
			}
		}
	}
	pthread_mutex_unlock(&session->mutex_obj);
}


/*
 * Purpose: get the connection by given connection_key, new if needed
 * Arguments: the corresponding session and the connection key
 * Returns: the target connection
 * Bug: set connection_type to TCP
 */
struct connection * get_connection(struct session * session, u_int16_t sport, u_int16_t dport)
{
	char connection_key[CONNECTION_KEY_LENGTH];
	struct connection * connection;

	memset(connection_key, 0, CONNECTION_KEY_LENGTH);
	sprintf(connection_key, "tcp_%s:%d_%s:%d", session->sip, sport, session->dip, dport);

	if((connection = hash_map_get(connection_key)) == NULL) {
		/* init a new connection */
		if((connection = (struct connection *) malloc(sizeof(struct connection))) == NULL) { warn("get_connection: malloc"); return NULL; }
		memset(connection, 0, sizeof(struct connection));
		connection->connection_type = CONNECTION_TYPE_TCP;
		strncpy(connection->key, connection_key, CONNECTION_KEY_LENGTH - 1);
		connection->session = session;
		connection->sport = sport;
		connection->dport = dport;
		if(pthread_mutex_init(&connection->mutex_obj, NULL) != 0) { warnx("get_connection: pthread_mutex_init"); return NULL; }
		if(pthread_cond_init(&connection->cond_obj, NULL) != 0) { warnx("get_connection: pthread_cont_init"); return NULL; }

		/* insert into queue */
		{
			struct connection * tail;

			pthread_mutex_lock(&session->mutex_obj);
			for(tail=session->connections; tail!=NULL && tail->next!=NULL; tail=tail->next) ;
			if(tail == NULL) session->connections = connection;
			else tail->next = connection;
			pthread_mutex_unlock(&session->mutex_obj);
		}
		if(hash_map_put(connection_key, connection) == -1) { warnx("get_connection: hash_map_put"); return NULL; }
		warnx("A new connection [%s]", connection_key);

		if(config.blacklist_on && blacklist_module(session->dip, dport)) { start_change_path(session); }

		/* statistic */
		{
			config.connection_count++;
			if(session->status == SESSION_PATH_CHANGED_STATUS) {
				config.relayed_connection_count++;
				config.drop_connection_count++;

				if(!config.relay_timestamp_on) {
					config.relay_timestamp_on = 1;
					if(gettimeofday(&config.relay_timestamp, NULL) == -1) { warn("gettimeofday"); }
				}
			}
		}

		pthread_mutex_lock(&session->mutex_obj);
		if(session->status == SESSION_PATH_CHANGED_STATUS) { connection->status = CONNECTION_RELAY_STATUS; }
		pthread_mutex_unlock(&session->mutex_obj);

		pthread_cond_signal(&session->cond_obj);
	}

	return connection;
}

/*
 * Purpose: queue packet
 * Arguments: packet and length
 * Returns: none
 */
struct packet * get_packet(struct connection * connection, char * packet, u_int32_t packet_len)
{
	struct packet * new_packet, * tail;

	if((new_packet = (struct packet *) malloc(sizeof(struct packet))) == NULL) { err(1, "malloc"); return NULL; }
	memset(new_packet, 0, sizeof(struct packet));
	memcpy(new_packet->data, packet, MAX_PACKET_SIZE);
	new_packet->len = packet_len;
	new_packet->connection = connection;

	pthread_mutex_lock(&connection->mutex_obj);
	if(connection->status == CONNECTION_REDIRECT_STATUS) { new_packet->status = PACKET_RESPONSE_WRITEBACK_STATUS; }
	for(tail=connection->packets; tail!=NULL && tail->next!=NULL; tail=tail->next) ;
	if(tail == NULL) connection->packets = new_packet;
	else tail->next = new_packet;
	pthread_mutex_unlock(&connection->mutex_obj);

	pthread_cond_signal(&connection->cond_obj);    /* There should only one thread wait */

	return new_packet;

	/* dump libpcap packet */
	//  if(config.queued_packets_pcap_dump_on) { pcap_dumper_dump(&config.queued_packets_pcap_dumper, packet, packet_len); }
}

/*
 * Purpose: start routine for receive packet from redirection interface
 * Arguments: corresponding session and target ip
 * Returns: none
 */
void start_change_path(struct session * session)
{
	if(session->status == SESSION_PATH_CHANGED_STATUS || session->status == SESSION_ALERTED) { return; }

	session->status = SESSION_ALERTED;

	{
		struct timeval gg;

		memset(&gg, 0, sizeof(struct timeval));
		gettimeofday(&gg, NULL);
		warnx("%ld.%ld SET_ALTER", gg.tv_sec, gg.tv_usec);
	}

	/* statistic */
	config.drop_session_count++;
	if(!config.redirect_timestamp_on) {
		config.redirect_timestamp_on = 1;
		if(gettimeofday(&config.redirect_timestamp, NULL) == -1) { warn("start_change_path: gettimeofday"); }
	}

	if(!config.decoy_if_on) { warnx("decoys are not open"); return; }
	if(save_memory) { return; }

	if((session->decoy = get_decoy()) == NULL) { save_memory = 1; warnx("no decoy available"); return; }

	/* statistic and set status, high cost, should execute after confirm to change path, i.e. no following flow exception */
	{
		struct connection * connection;

		config.replay_session_count++;

		for(connection=session->connections; connection!=NULL; connection=connection->next) {
			struct packet * packet;
			int dead_flag;
			u_int64_t byte_count, packet_count;

			dead_flag = 0;
			byte_count = packet_count = 0;
			for(packet=connection->packets; packet!=NULL; packet=packet->next) {
				struct packet_attribute attr;

				byte_count += packet->len;
				packet_count++;

				attr = packet_parse(packet->data);
				if(attr.ip && attr.tcp) {
					/* Bug: Ignore same 5 tuple but new connection */
					if(attr.tcp_header->fin || attr.tcp_header->rst) dead_flag = 1;
				}
			}
			if(dead_flag) {
				config.replayed_connection_count++;
				config.replayed_packet_count += packet_count;
				config.replayed_byte_count += byte_count;
				connection->status = CONNECTION_REPLAY_STATUS;
			} else {
				config.redirected_connection_count++;
				config.redirected_packet_count += packet_count;
				config.redirected_byte_count += byte_count;
				connection->status = CONNECTION_REDIRECT_STATUS;
			}
		}
		session->status = SESSION_PATH_CHANGED_STATUS;
	}

	/* create a never die pthread */
	if(pthread_create(&session->thread_obj, NULL, pthread_start_change_path, session) != 0) { err(1, "pthread_create"); }
	warnx("change network path for session: %s", session->key);
}

/*
 * Purpose: pthread start routine for receive packet from redirection interface
 * Arguments: struct session
 * Returns: none
 * Bug: receive raw socket maybe fail in sll_pkttype setting
 */
void * pthread_start_change_path(void * arg)
{
	struct session * session;
	struct connection * connection;
	int wait_flag;

	session = (struct session *) arg;
	wait_flag = 0;

next_round:
	wait_flag++;

	for(connection=session->connections; connection!=NULL; connection=connection->next) {
		pthread_attr_t attr;

		if(connection->handle_by_thread_flag) continue;

		wait_flag = 0;
		connection->handle_by_thread_flag = 1;
		if(pthread_attr_init(&attr) != 0) { warnx("pthread_start_change_path: pthread_attr_init"); }
		if(pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED) != 0) { warnx("pthread_start_change_path: pthread_attr_setdetachstate"); }
		if(pthread_attr_destroy(&attr) != 0) { warnx("pthread_start_change_path: pthread_attr_destroy"); }
		switch(connection->status) {
			case CONNECTION_REPLAY_STATUS:
				if(pthread_create(&connection->thread_obj, &attr, pthread_start_replay, connection) != 0) { warnx("pthread_start_change_path: pthread_create error"); }
				break;
			case CONNECTION_REDIRECT_STATUS:
				if(pthread_create(&connection->thread_obj, &attr, pthread_start_redirect, connection) != 0) { warnx("pthread_start_change_path: pthread_create error"); }
				break;
			case CONNECTION_RELAY_STATUS:
				if(pthread_create(&connection->thread_obj, &attr, pthread_start_relay, connection) != 0) { warnx("pthread_start_change_path: pthread_create error"); }
				break;
			default:
				warnx("pthread_start_change_path: it should never happen");
				break;
		}
	}

	if(wait_flag > 5) {
		pthread_mutex_lock(&session->mutex_obj);
	warnx("wait..");
		pthread_cond_wait(&session->cond_obj, &session->mutex_obj);
		pthread_mutex_unlock(&session->mutex_obj);
	}
	goto next_round;

	/* actually, unreachable */
	warnx("pthread_start_change_path die");

	pthread_exit(NULL);
	return NULL;
}

void * pthread_start_replay(void * arg)
{
	struct connection * connection;
	struct packet * packet;
	struct packet_attribute attr;
	struct sockaddr_in addr;
	int fd, n;

	warnx("pthread_start_replay start");

	connection = (struct connection *) arg;

	{
		char sbuf[1024];
		static int ddn = 1;

		memset(sbuf, 0, 1024);
		sprintf(sbuf, "replay_%d.pcap", ddn++);
		strncpy(connection->pd.pcap_dump_filename, sbuf, FILENAME_LENGTH);
		if(!pcap_dumper_init(&connection->pd)) { warnx("pthread_start_replay: pcap_dumper_init"); }
	}


	if((fd = socket(AF_INET, SOCK_STREAM, 0)) == -1) { err(1, "pthread_start_replay: socket"); }
	memset(&addr, 0, sizeof(struct sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_port = htons(connection->dport);
	addr.sin_addr.s_addr = connection->session->decoy->ip;
	if(connect(fd, (struct sockaddr *) &addr, sizeof(struct sockaddr_in)) == -1) { warn("pthread_start_replay: bind"); goto replay_end; }

	connection = (struct connection *) arg;
	for(packet=connection->packets; packet!=NULL; packet=packet->next) {
		attr = packet_parse(packet->data);

		if(attr.ip && attr.tcp) {
			if((n = write(fd, ((char *) attr.tcp_header) + attr.tcp_header_len, attr.tcp_data_len)) == -1) { warn("pthread_start_replay: write"); goto replay_end; }
			else if(n != attr.tcp_data_len) { warn("pthread_start_replay: write length not match"); }
			usleep(3);

			{
				pcap_dumper_dump(&connection->pd, packet->data, packet->len);
			}
		}
	}

replay_end:
	remove_connection(connection->session, connection);

	free(connection);
	close(fd);

	pthread_exit(NULL);
	return NULL;
}

void * pthread_start_redirect(void * arg)
{
	struct connection * connection;
	int wfd, rfd, wbfd=1;
	u_int32_t sip;
	u_int16_t sport, dport;

	warnx("pthread_start_redirect start");

	connection = (struct connection *) arg;

	{
		char sbuf[1024];
		static int ddn = 1;

		memset(sbuf, 0, 1024);
		sprintf(sbuf, "redirect_%d.pcap", ddn++);
		strncpy(connection->pd.pcap_dump_filename, sbuf, FILENAME_LENGTH);
		if(!pcap_dumper_init(&connection->pd)) { warnx("pthread_start_redirect: pcap_dumper_init"); }
	}

	if(inet_pton(AF_INET, connection->session->sip, &sip) == -1) { err(1, "inet_pton"); }
	sport = htons(connection->sport);
	dport = htons(connection->dport);

	/* send raw socket */
	{
		struct sockaddr_ll addr;
		socklen_t addr_len;

		if((wfd = socket(AF_PACKET, SOCK_RAW, htons(ETH_P_ALL))) == -1) { err(1, "socket"); }

		memset(&addr, 0, sizeof(struct sockaddr_ll));
		addr.sll_family = AF_PACKET;
		memcpy((char *) &addr.sll_addr, (char *) &connection->session->decoy->mac, 6);
		addr.sll_halen = 6;
		addr.sll_ifindex = config.decoy_if_index;

		addr_len = sizeof(struct sockaddr_ll);

		if(bind(wfd, (struct sockaddr *) &addr, addr_len) == -1) { err(1, "bind"); }
	}

	/* receive raw socket */
	{
		struct sockaddr_ll addr;
		socklen_t addr_len;

		if((rfd = socket(AF_PACKET, SOCK_RAW, htons(ETH_P_ALL))) == -1) { err(1, "socket"); }

		memset(&addr, 0, sizeof(struct sockaddr_ll));
		addr.sll_family = AF_PACKET;
		memcpy((char *) &addr.sll_addr, (char *) &connection->session->decoy->mac, 6);
		addr.sll_halen = 6;
		addr.sll_ifindex = config.decoy_if_index;

		/* note this config */
		addr.sll_pkttype = PACKET_OTHERHOST;
		addr.sll_protocol = htons(ETH_P_ALL);

		addr_len = sizeof(struct sockaddr_ll);

		if(bind(rfd, (struct sockaddr *) &addr, addr_len) == -1) { err(1, "bind"); }
	}

	/* write back send raw socket */
	if(config.bot_if_on) {
		struct sockaddr_ll addr;
		socklen_t addr_len;

		if((wbfd = socket(AF_PACKET, SOCK_RAW, htons(ETH_P_ALL))) == -1) { err(1, "socket"); }

		memset(&addr, 0, sizeof(struct sockaddr_ll));
		addr.sll_family = AF_PACKET;
		memcpy((char *) &addr.sll_addr, (char *) config.bot_mac.ether_addr_octet, 6);
		addr.sll_halen = 6;
		addr.sll_ifindex = config.bot_if_index;

		addr_len = sizeof(struct sockaddr_ll);

		if(bind(wbfd, (struct sockaddr *) &addr, addr_len) == -1) { err(1, "bind"); }
	}

	{
		char receive_packet[BUFSIZ];
		struct packet_attribute send_packet_attr, receive_packet_attr;
		struct packet * packet;
		int wait_flag, ret, len;

		wait_flag = 0;

next_write:
		if((packet = connection->packets) != NULL) {
			wait_flag = 0;

			pthread_mutex_lock(&connection->mutex_obj);
			connection->packets = connection->packets->next;
			pthread_mutex_unlock(&connection->mutex_obj);

			send_packet_attr = packet_parse(packet->data);

			layer_stateful_modules(connection, send_packet_attr, 7, OP_WRITE);
			layer_stateful_modules(connection, send_packet_attr, 5, OP_WRITE);
			layer_stateful_modules(connection, send_packet_attr, 4, OP_WRITE);
			layer_stateful_modules(connection, send_packet_attr, 3, OP_WRITE);
			layer_stateful_modules(connection, send_packet_attr, 2, OP_WRITE);
			layer_stateful_modules(connection, send_packet_attr, 1, OP_WRITE);

			if(packet->len > 1500) { warnx("[redirect] packet length too long %d", packet->len); goto free_packet; }
			if((ret = write(wfd, packet->data, packet->len)) == -1) { err(1, "write"); }
			else if(ret != packet->len) { warnx("[redirect] write length not match"); }
			warnx("[redirect] write one");

			{
				pcap_dumper_dump(&connection->pd, packet->data, packet->len);
			}

next_read:
			memset(receive_packet, 0, BUFSIZ);
			if((ret = read(rfd, receive_packet, BUFSIZ - 1)) == -1) { err(1, "pthread_start_redirect: read"); }
			else if(ret == 0) { warnx("pthread_start_redirect read end?"); goto redirect_end; }

			receive_packet_attr = packet_parse(receive_packet);

			if(memcmp(receive_packet_attr.ether_header->h_dest, &(connection->session->decoy->mac), 6) == 0) { goto next_read;; }    /* our program write out */
			if(!receive_packet_attr.ip || !receive_packet_attr.tcp) { goto next_read;; }

			/* check 5 tuple */
			if(receive_packet_attr.ip_header->saddr != connection->session->decoy->ip || receive_packet_attr.ip_header->daddr != sip ||
					receive_packet_attr.tcp_header->source != dport || receive_packet_attr.tcp_header->dest != sport) { goto next_read;; } 

			layer_stateful_modules(connection, receive_packet_attr, 7, OP_READ);
			layer_stateful_modules(connection, receive_packet_attr, 5, OP_READ);
			layer_stateful_modules(connection, receive_packet_attr, 4, OP_READ);
			layer_stateful_modules(connection, receive_packet_attr, 3, OP_READ);
			layer_stateful_modules(connection, receive_packet_attr, 2, OP_READ);
			layer_stateful_modules(connection, receive_packet_attr, 1, OP_READ);

			{
				pcap_dumper_dump(&connection->pd, receive_packet, ret);
			}

			/* write back */
			if(packet->status == PACKET_RESPONSE_WRITEBACK_STATUS) {
				layer_stateful_modules(connection, receive_packet_attr, 7, OP_WRITEBACK);
				layer_stateful_modules(connection, receive_packet_attr, 5, OP_WRITEBACK);
				layer_stateful_modules(connection, receive_packet_attr, 4, OP_WRITEBACK);
				layer_stateful_modules(connection, receive_packet_attr, 3, OP_WRITEBACK);
				layer_stateful_modules(connection, receive_packet_attr, 2, OP_WRITEBACK);
				layer_stateful_modules(connection, receive_packet_attr, 1, OP_WRITEBACK);

				if(ret > 1500) { warnx("[redirect] packet length too long %d", ret); goto free_packet; }
				if((len = write(wbfd, receive_packet, ret)) == -1) { err(1, "write"); }
				else if(len != ret) { warnx("[redirect] write length not match"); }

				warnx("[redirect %s] write back one", connection->key);
			}

free_packet:
			free(packet);
		} else {
			wait_flag++;
		}

		if(wait_flag > 5) {
			pthread_mutex_lock(&connection->mutex_obj);
			pthread_cond_wait(&connection->cond_obj, &connection->mutex_obj);
			pthread_mutex_unlock(&connection->mutex_obj);
		}

		goto next_write;
	}

redirect_end:
	pthread_exit(NULL);
	return NULL;
}

void * pthread_start_relay(void * arg)
{
	struct connection * connection;
	int wfd;
	u_int8_t wait_flag;

	warnx("pthread_start_relay start");

	connection = (struct connection *) arg;

	{
		char sbuf[1024];
		static int ddn = 1;

		memset(sbuf, 0, 1024);
		sprintf(sbuf, "relay_%d.pcap", ddn++);
		strncpy(connection->pd.pcap_dump_filename, sbuf, FILENAME_LENGTH);
		if(!pcap_dumper_init(&connection->pd)) { warnx("pthread_start_redirect: pcap_dumper_init"); }
	}

	warnx("[relay] connection %s", connection->key);

	{
		pthread_attr_t attr;

		if(pthread_attr_init(&attr) != 0) { warnx("pthread_start_redirect: pthread_attr_init"); }
		if(pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED) != 0) { warnx("pthread_start_redirect: pthread_attr_setdetachstate"); }
		if(pthread_create(&connection->thread_obj2, &attr, pthread_start_relay_receive, connection) != 0) { err(1, "pthread_create"); }
		if(pthread_attr_destroy(&attr) != 0) { warnx("pthread_start_redirect: pthread_attr_destroy"); }
	}

	/* send raw socket */
	{
		struct sockaddr_ll addr;
		socklen_t addr_len;

		if((wfd = socket(AF_PACKET, SOCK_RAW, htons(ETH_P_ALL))) == -1) { err(1, "socket"); }

		memset(&addr, 0, sizeof(struct sockaddr_ll));
		addr.sll_family = AF_PACKET;
		memcpy((char *) &addr.sll_addr, (char *) &connection->session->decoy->mac, 6);
		addr.sll_halen = 6;
		addr.sll_ifindex = config.decoy_if_index;

		addr_len = sizeof(struct sockaddr_ll);

		if(bind(wfd, (struct sockaddr *) &addr, addr_len) == -1) { err(1, "bind"); }
	}

	wait_flag = 0;
	while(1) {
		struct packet * send_packet = NULL;
		int ret;

		if((send_packet = connection->packets) != NULL) {
			struct packet_attribute send_packet_attr;

			pthread_mutex_lock(&connection->session->mutex_obj);
			connection->packets = connection->packets->next;
			pthread_mutex_unlock(&connection->session->mutex_obj);

			wait_flag = 0;

			send_packet_attr = packet_parse(send_packet->data);

			layer_stateful_modules(connection, send_packet_attr, 7, OP_WRITE);
//			layer_stateful_modules(connection, send_packet_attr, 5, OP_WRITE);
			layer_stateful_modules(connection, send_packet_attr, 3, OP_WRITE);
			layer_stateful_modules(connection, send_packet_attr, 2, OP_WRITE);
			layer_stateful_modules(connection, send_packet_attr, 1, OP_WRITE);

			if(send_packet->len > 1500) { warnx("[relay] packet length too long %d", send_packet->len); goto free_packet; }
			if((ret = write(wfd, send_packet->data, send_packet->len)) == -1) { err(1, "write"); }
			else if(ret != send_packet->len) { warnx("[relay] write length not match"); }
			warnx("[relay %s] write one", connection->key);

			{
				pcap_dumper_dump(&connection->pd, send_packet->data, send_packet->len);
			}

free_packet:
			;
//			free(send_packet);
		}

		if(wait_flag > 5) {
			pthread_mutex_lock(&connection->mutex_obj);
			pthread_cond_wait(&connection->cond_obj, &connection->mutex_obj);
			pthread_mutex_unlock(&connection->mutex_obj);
		}

		wait_flag++;
	}

	pthread_exit(NULL);
	return NULL;
}

void * pthread_start_relay_receive(void * arg)
{
	int rfd, wbfd=1;
	struct connection * connection;
	u_int32_t sip;
	u_int16_t sport, dport;

	warnx("pthread_start_relay_receive start");

	connection = (struct connection *) arg;

	if(inet_pton(AF_INET, connection->session->sip, &sip) == -1) { err(1, "inet_pton"); }
	sport = htons(connection->sport);
	dport = htons(connection->dport);

	/* receive raw socket */
	{
		struct sockaddr_ll addr;
		socklen_t addr_len;

		if((rfd = socket(AF_PACKET, SOCK_RAW, htons(ETH_P_ALL))) == -1) { err(1, "socket"); }

		memset(&addr, 0, sizeof(struct sockaddr_ll));
		addr.sll_family = AF_PACKET;
		memcpy((char *) &addr.sll_addr, (char *) &(connection->session->decoy->mac), 6);
		addr.sll_halen = 6;
		addr.sll_ifindex = config.decoy_if_index;

		/* note this config */
		addr.sll_pkttype = PACKET_OTHERHOST;
		addr.sll_protocol = htons(ETH_P_ALL);

		addr_len = sizeof(struct sockaddr_ll);

		if(bind(rfd, (struct sockaddr *) &addr, addr_len) == -1) { err(1, "bind"); }
	}

	/* write back send raw socket */
	if(config.bot_if_on) {
		struct sockaddr_ll addr;
		socklen_t addr_len;

		if((wbfd = socket(AF_PACKET, SOCK_RAW, htons(ETH_P_ALL))) == -1) { err(1, "socket"); }

		memset(&addr, 0, sizeof(struct sockaddr_ll));
		addr.sll_family = AF_PACKET;
		memcpy(&addr.sll_addr, &config.bot_mac, 6);
		addr.sll_halen = 6;
		addr.sll_ifindex = config.bot_if_index;

		addr_len = sizeof(struct sockaddr_ll);

		if(bind(wbfd, (struct sockaddr *) &addr, addr_len) == -1) { err(1, "bind"); }
	}

	while(1) {
		char receive_packet[BUFSIZ];
		struct packet_attribute receive_packet_attr;
		int len, ret;

		memset(receive_packet, 0, BUFSIZ);
		if((ret = read(rfd, receive_packet, BUFSIZ-1)) == -1) { err(1, "read"); }
		receive_packet_attr = packet_parse(receive_packet);

		if(memcmp(receive_packet_attr.ether_header->h_dest, &(connection->session->decoy->mac), 6) == 0) { continue; }    /* our program write out */
		if(!receive_packet_attr.ip || !receive_packet_attr.tcp) { continue; }

		/* check 5 tuple */
		if(receive_packet_attr.ip_header->saddr != connection->session->decoy->ip || receive_packet_attr.ip_header->daddr != sip ||
				receive_packet_attr.tcp_header->source != dport || receive_packet_attr.tcp_header->dest != sport) { continue; } 

		layer_stateful_modules(connection, receive_packet_attr, 7, OP_READ);
//		layer_stateful_modules(connection, receive_packet_attr, 5, OP_READ);
		layer_stateful_modules(connection, receive_packet_attr, 3, OP_READ);
		layer_stateful_modules(connection, receive_packet_attr, 2, OP_READ);
		layer_stateful_modules(connection, receive_packet_attr, 1, OP_READ);

		warnx("[relay %s] read one", connection->key);

		{
			pcap_dumper_dump(&connection->pd, receive_packet, ret);
		}

		/* write back */
		{
			layer_stateful_modules(connection, receive_packet_attr, 7, OP_WRITEBACK);
//			layer_stateful_modules(connection, receive_packet_attr, 5, OP_WRITEBACK);
			layer_stateful_modules(connection, receive_packet_attr, 3, OP_WRITEBACK);
			layer_stateful_modules(connection, receive_packet_attr, 2, OP_WRITEBACK);
			layer_stateful_modules(connection, receive_packet_attr, 1, OP_WRITEBACK);

			if(ret > 1500) { warnx("[relay] packet length too long %d", ret); continue; }
			if((len = write(wbfd, receive_packet, ret)) == -1) { err(1, "write"); }
			else if(len != ret) { warnx("[relay] write length not match"); }

			warnx("[relay %s] write back one", connection->key);
		}
	}

	pthread_exit(NULL);
	return NULL;
}

u_int32_t get_verdict(struct packet * packet)
{
	struct timeval gg;
	struct packet_attribute attr;

	memset(&gg, 0, sizeof(struct timeval));
	gettimeofday(&gg, NULL);

	attr = packet_parse(packet->data);

	if(packet->connection->session->status == SESSION_PATH_CHANGED_STATUS || packet->connection->session->status == SESSION_ALERTED) {

		if(attr.tcp && packet->connection != NULL && packet->connection->session) {
			warnx("DROP %ld.%ld sip=%s dip=%s sport=%d dport=%d seq=%u",
					gg.tv_sec, gg.tv_usec,
					packet->connection->session->sip, packet->connection->session->dip,
					ntohs(attr.tcp_header->source), ntohs(attr.tcp_header->dest), ntohl(attr.tcp_header->seq));
		}

		return NF_DROP;
	} else {

		if(attr.tcp && packet->connection != NULL && packet->connection->session) {
			warnx("ACCEPT %ld.%ld sip=%s dip=%s sport=%d dport=%d seq=%u",
					gg.tv_sec, gg.tv_usec,
					packet->connection->session->sip, packet->connection->session->dip,
					ntohs(attr.tcp_header->source), ntohs(attr.tcp_header->dest), ntohl(attr.tcp_header->seq));
		}

		return config.verdict;
	}
}
